/* -*- C++ -*-
 *
 * <<< vision_control_unit.h >>>
 *
 * --- control class 'vision_control_unit'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef VISION_CONTROL_UNIT_H
#define VISION_CONTROL_UNIT_H 1

#include <string>
#include <iostream>
#include <isis/synchronous_unit.h>
#include "transmit_unit.h"
#include "transmit_status_buffer.h"
#include "input_unit.h"
#include "result_unit.h"
#include "receive_reg.h"
#include "pico_memory_map.h"

//#define TU_DEBUG
//#define UP_SEND_D

template <class A, class D = A, size_t Dsz = sizeof(D), class Ch = char >
class vision_control_unit : public synchronous_unit
{
private:
	typedef vision_control_unit<A, D, Dsz, Ch> thisclass;

public:
	typedef A  address_type;
	typedef D  data_type;
	typedef Ch char_type;
	typedef A  size_type;
	static const size_t sizeof_data_type = Dsz;

private:
	static const int SIZE = 4;
	transmit_unit t_unit[SIZE]; 
	vision_input_unit *in_unit;
	vision_result_output_unit *ro_unit;

	data_type data_status_bit_;
	address_type address_buffer_;
	data_type data_buffer_;

	transmit_status_buffer *transmit_status_buffer_;

	int controler_number_;

	enum {
		UP,
		DOWN,
		RIGHT,
		LEFT,
	};

	controler_state state;

	receive_reg_ receive_reg;

public:
	vision_control_unit(void);
	vision_control_unit(const thisclass&){};
	vision_control_unit(int a){};
	~vision_control_unit(){};
	
	/* read & set status */

	/* Access Buffer */ 
	data_type& data_buffer(void) { return data_buffer_; }
	data_type& address_buffer(void) { return address_buffer_; }
	int& controler_number() { return controler_number_; }
	data_type get_transmit_status() { return transmit_status_buffer_ -> get_status(); }

	/* Access Transmit Unit Register Address */
	data_type left_r_d_reg() { return t_unit[LEFT].data_receive_latch(); }
	data_type right_r_d_reg() { return t_unit[RIGHT].data_receive_latch(); }
	data_type up_r_d_reg() { return t_unit[UP].data_receive_latch(); }
	data_type down_r_d_reg() { return t_unit[DOWN].data_receive_latch(); }

	data_type left_r_s_reg() { return t_unit[LEFT].status_receive_latch(); }
	data_type right_r_s_reg() { return t_unit[RIGHT].status_receive_latch(); }
	data_type up_r_s_reg() { return t_unit[UP].status_receive_latch(); }
	data_type down_r_s_reg(){ return t_unit[DOWN].status_receive_latch(); }
	data_type input_data() { return in_unit->data_buffer(); }
	void input_decode0() { in_unit -> decode0() = false; }
	void input_decode1() { in_unit -> decode1() = false; }

	/* Function */
	void reset(){};

	/* Clock */
	void clock_in();
	void clock_out();

	/* Port Connection */
	vision_port& left_data_input() { return t_unit[RIGHT].input_data_port(); }
	vision_port& left_status_input() { return t_unit[RIGHT].input_status_port(); }
	vision_port& right_data_input() {return t_unit[LEFT].input_data_port(); };
	vision_port& right_status_input() {return t_unit[LEFT].input_status_port(); };

	vision_port& left_data_output() { return t_unit[LEFT].output_data_port(); };

	vision_port& left_status_output() { return t_unit[LEFT].output_status_port(); };
	vision_port& right_data_output() { return t_unit[RIGHT].output_data_port(); }
	vision_port& right_status_output() { return t_unit[RIGHT].output_status_port(); };

	vision_port& left_input_send(){ return t_unit[RIGHT].input_send_port();};
	vision_port& left_input_receive(){ return t_unit[LEFT].input_receive_port();};
	vision_port& left_output_send(){ return t_unit[LEFT].output_send_port();};
	vision_port& left_output_receive(){ return t_unit[RIGHT].output_receive_port();};

	vision_port& right_input_send(){ return t_unit[LEFT].input_send_port(); };
	vision_port& right_input_receive(){ return t_unit[RIGHT].input_receive_port(); };
	vision_port& right_output_send(){ return t_unit[RIGHT].output_send_port(); };
	vision_port& right_output_receive(){ return t_unit[LEFT].output_receive_port(); };

	vision_port& up_data_input() { return t_unit[DOWN].input_data_port(); }
	vision_port& up_status_input() { return t_unit[DOWN].input_status_port(); }
	vision_port& down_data_input() {return t_unit[UP].input_data_port(); };
	vision_port& down_status_input() {return t_unit[UP].input_status_port(); };

	vision_port& up_data_output() { return t_unit[UP].output_data_port(); };

	vision_port& up_status_output() { return t_unit[UP].output_status_port(); };
	vision_port& down_data_output() { return t_unit[DOWN].output_data_port(); }
	vision_port& down_status_output() { return t_unit[DOWN].output_status_port(); };

	vision_port& up_input_send(){ return t_unit[DOWN].input_send_port();};
	vision_port& up_input_receive(){ return t_unit[UP].input_receive_port();};
	vision_port& up_output_send(){ return t_unit[UP].output_send_port();};
	vision_port& up_output_receive(){ return t_unit[DOWN].output_receive_port();};

	vision_port& down_input_send(){ return t_unit[UP].input_send_port(); };
	vision_port& down_input_receive(){ return t_unit[DOWN].input_receive_port(); };
	vision_port& down_output_send(){ return t_unit[DOWN].output_send_port(); };
	vision_port& down_output_receive(){ return t_unit[UP].output_receive_port(); };

	vision_port& before_input() { return in_unit->Data_Input_Port(); }
	vision_port& result_output() { return ro_unit->Data_Output_Port(); }
	vision_input_unit& front_input() { return *in_unit; }

	//for debug
	receive_reg_& receive_data(void) {
		return receive_reg;
	}

};


template<class A, class D, size_t Dsz, class Ch>
inline void vision_control_unit<A, D, Dsz, Ch>::clock_in(void)
{

	/* set & read data buffer */
	/* store operation */

	switch(address_buffer_){
	case RESULT:
		ro_unit -> data_buffer() = data_buffer_;
		//ro_unit -> result_status() = 1;
		ro_unit -> decode0() = true;
		break;
	case RESULT_CONTROL:
		ro_unit -> decode1() = true;
		break;
	case UP_SEND_DATA:
#ifdef UP_SEND_D
		cout << "UP SEND DATA " << endl;
#endif
		t_unit[UP].data_transmit_register() = data_buffer_;

		break;
	case DOWN_SEND_DATA:
		t_unit[DOWN].data_transmit_register() = data_buffer_;
	
		break;
	case RIGHT_SEND_DATA:
		t_unit[RIGHT].data_transmit_register() = data_buffer_;

		break;
	case LEFT_SEND_DATA:
		t_unit[LEFT].data_transmit_register() = data_buffer_;

		break;
	case UP_SEND_STATUS:
#ifdef UP_SEND_D
		cout << "UP SEND STATUS" << endl;
#endif
		t_unit[UP].status_transmit_register() = data_buffer_;
		t_unit[UP].send_status() = 1; 
		break;
	case DOWN_SEND_STATUS:
		t_unit[DOWN].status_transmit_register() = data_buffer_;
		t_unit[DOWN].send_status() = 1; 

		break;
	case RIGHT_SEND_STATUS:
		t_unit[RIGHT].status_transmit_register() = data_buffer_;
		t_unit[RIGHT].send_status() = 1; 

		break;
	case LEFT_SEND_STATUS:
		t_unit[LEFT].status_transmit_register() = data_buffer_;
		t_unit[LEFT].send_status() = 1; 

		break;
	case TRANSMIT_STATUS:
		data_buffer_ = transmit_status_buffer_ -> get_status();

		break;
	case READY:

		break;
	default:

		break;
	}	

	/* transmit unit clock in */
//	cout << "controler clock_in : " << address_buffer_ ;
//	cout << " : " << data_buffer_ << endl;

	t_unit[DOWN].clock_in();
	t_unit[UP].clock_in();
	t_unit[LEFT].clock_in();
	t_unit[RIGHT].clock_in();
	in_unit -> clock_in();
	ro_unit -> clock_in();

}

template<class A, class D, size_t Dsz, class Ch>
inline void vision_control_unit<A, D, Dsz, Ch>::clock_out(void)
{
	/* transmit unit clock out */
//	cout << "controler clock_out : " << address_buffer_ ;
//	cout << " : " << data_buffer_ << endl;

	/* set status */
	switch(address_buffer_){
	case UP_RECEIVE_STATUS:
		t_unit[UP].output_receive() = 0;
		break;
	case DOWN_RECEIVE_STATUS:
		t_unit[DOWN].output_receive() = 0;
		break;
	case RIGHT_RECEIVE_STATUS:
		t_unit[RIGHT].output_receive() = 0;
		break;
	case LEFT_RECEIVE_STATUS:
		t_unit[LEFT].output_receive() = 0;
		break;
	default:
		break;
	}

	t_unit[DOWN].clock_out();
	t_unit[UP].clock_out();
	t_unit[LEFT].clock_out();
	t_unit[RIGHT].clock_out();
	ro_unit -> clock_out();
	in_unit -> clock_out();

	/* set transmit status buffer */
	transmit_status_buffer_ -> left_receive(t_unit[RIGHT].output_receive());
	transmit_status_buffer_ -> right_receive(t_unit[LEFT].output_receive());
	transmit_status_buffer_ -> up_receive(t_unit[DOWN].output_receive());
	transmit_status_buffer_ -> down_receive(t_unit[UP].output_receive());

/*
	transmit_status_buffer_ -> left_send(t_unit[LEFT].output_send());
	transmit_status_buffer_ -> right_send(t_unit[RIGHT].output_send());
	transmit_status_buffer_ -> up_send(t_unit[UP].output_send());
	transmit_status_buffer_ -> down_send(t_unit[DOWN].output_send());
	*/

	transmit_status_buffer_ -> left_send(t_unit[LEFT].send_status() 
			|| t_unit[LEFT].transmit_status());
	transmit_status_buffer_ -> right_send(t_unit[RIGHT].send_status()
			|| t_unit[RIGHT].transmit_status());
	transmit_status_buffer_ -> up_send(t_unit[UP].send_status()
			|| t_unit[UP].transmit_status());
	transmit_status_buffer_ -> down_send(t_unit[DOWN].send_status()
			|| t_unit[DOWN].transmit_status());
	transmit_status_buffer_ -> input_status(in_unit -> input_status());
	transmit_status_buffer_ -> result_status(ro_unit -> result_status());

	address_buffer_ = READY;

	//set debug data
	receive_reg.left_data = left_r_d_reg();
	receive_reg.right_data = right_r_d_reg();
	receive_reg.up_data = up_r_d_reg();
	receive_reg.down_data = down_r_d_reg();
	receive_reg.left_status = left_r_s_reg();
	receive_reg.right_status = right_r_s_reg();
	receive_reg.up_status = up_r_s_reg();
	receive_reg.down_status = down_r_s_reg();

}

template <class A, class D, size_t Dsz, class Ch>
vision_control_unit<A, D, Dsz, Ch>::vision_control_unit()
	: in_unit(new vision_input_unit()),
	  ro_unit(new vision_result_output_unit()),
	  transmit_status_buffer_(new transmit_status_buffer()),
	  state(READY),
	  address_buffer_(READY)

{
}

//ostream& operator<<(ostream&, const vision_control_unit&);



#endif // VISION_CONTROL_UNIT_H 
