/*
 * <<< pico_instruction.cc >>>
 *
 * --- PICO instruction class 'pico_instruction'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstring> // needed only for SGI C++ compiler
#include <string>
#include <iomanip> // needed only for SGI C++ compiler
#include <iostream>
#include <strstream>
#include "pico_instruction.h"
#include "pico_register_file.h"

using namespace std;

template <class T>
static string to_string(T x)
{
	strstream buf;
	T tmp;
	if (x > 0 || x == 0) { // erase warning
		tmp = x;
	} else {
		buf << '-';
		tmp = -x;
	}

	if (tmp == 0 || ((tmp & 0xff) != 0 && ((tmp % 1000) == 0 || tmp < 1000))) {
		buf << hex << "0x" << tmp;
	} else if((tmp & 0x7ff) != 0) {
		buf << hex << "0x" << tmp;
	}
	else {
		buf << hex << "0x" << tmp;
	}
	
	string str;
	buf >> str;
	return str;
}

pico_instruction::pico_instruction(void)
{
	clear();
}

pico_instruction::pico_instruction(const pico_instruction& a)
	: image_(a.image_),
	  pc_(a.pc_),
	  opcode_(a.opcode_),
	  operation_(a.operation_)
{

}

pico_instruction::~pico_instruction()
{

}


string pico_instruction::opcode_string(void) const
{
	/* same place with pico_instruction.h enum opcode_type */
	static const char * name[] = {
		"NOP", "ADD", "ADDI", "SUB", "SUBI", "AND", "ANDI", "OR",
		"ORI",
		"SL", "SR", "LD", "LDLI", "LDHI", "ST", "MV", "BEQZ", "BNEZ",
		"UNKNOWN", "RESERVED", "XORI", "JALR", "BMI", "BPL", "JAL",
		"JR", "JMP", "R_TYPE_OP", "XOR", "NOT", "LB", "SB", "RFI", 
		"EINT",
		"LIAR", "SIAR", "DINT", "MUL", "DIV", "MULFI", "MULFF", "HALT",
		"PUSH", "POP", "MVR2S", "MVS2R", "MVR2T", "MVT2R", "MVR2B",
		"MVB2R", "MVT2S", "MVB2S", "NEXT", "LOOK", "S_TYPE_OP",
		"CMP", "CMPI", "PFX", "CALL", "RTN", "PUSHS", "POPS", "MULI",
		"MULFFI", "MULFII", "SSP2REG", "REG2SSP",
	};
	string buf(name[opcode()]);
	return buf;
}

string pico_instruction::operation_string(void) const
{
	strstream buf;

	const int SP_REG = 9;
	const int TB_REG = 17;
	const int BB_REG = 25;

	switch(opcode()){
	/*
	case NOP: case RTN: case HALT:
		buf << opcode_string();
		break;
		*/

	case ADD: case SUB: case CMP: case MV:
	case MUL: case MULFI: case MULFF:
	case OR: case SL: case SR: case AND: case XOR:
		buf << opcode_string() << ' ' 
			<< pico_register_file::register_name(rd()) << ", "
			<< pico_register_file::register_name(rs());
		break;

	case ADDI: case SUBI: case ANDI: case XORI: case CMPI:
	case ORI:  case MULI: case MULFII: case MULFFI: 
	case LDLI:
		buf << opcode_string() << ' '
			<< pico_register_file::register_name(rd()) << ", #"
			<< to_string(signed_immediate());
		break;

	case NEXT: case LOOK:
		buf << opcode_string() << ' ' 
			<< pico_register_file::register_name(rd()) << ", "
			<< pico_register_file::register_name(rs()+SP_REG);
		break;

	case LD: 
		buf << opcode_string() << ' ' 
			<< pico_register_file::register_name(rd()) << ", ("
			<< pico_register_file::register_name(rs()) << ")" ;
		break;

	case POP:  
		buf << opcode_string() << ' ' 
			<< pico_register_file::register_name(rd()) << ", ("
			<< pico_register_file::register_name(rs()+SP_REG) << ")" ;
		break;

	case ST: 
		buf << opcode_string() << " (" 
			<< pico_register_file::register_name(rd()) << "), "
			<< pico_register_file::register_name(rs());
		break;

	case PUSH: 
		buf << opcode_string() << " (" 
			<< pico_register_file::register_name(rd() + SP_REG) << "), "
			<< pico_register_file::register_name(rs());
		break;

	case BEQZ: case BNEZ: case BMI: case BPL: case JMP:
	case CALL: 
		buf << opcode_string() << ' ' 
			<< to_string(signed_long_immediate());
		break;

	case MVS2R:
		buf << opcode_string() << ' '
			<< pico_register_file::register_name(rd()) << ", "
			<< pico_register_file::register_name(rs() + SP_REG);
		break;

	case MVR2S: 
		buf << opcode_string() << ' '
			<< pico_register_file::register_name(rd()+SP_REG) << ", "
			<< pico_register_file::register_name(rs());
		break;

	case MVR2T: 
		buf << opcode_string() << ' '
			<< pico_register_file::register_name(rd()+TB_REG) << ", "
			<< pico_register_file::register_name(rs());
		break;

	case MVT2R:
		buf << opcode_string() << ' '
			<< pico_register_file::register_name(rd()) << ", "
			<< pico_register_file::register_name(rs()+TB_REG);
		break;

	case MVR2B:
		buf << opcode_string() << ' '
			<< pico_register_file::register_name(rd()+BB_REG) << ", "
			<< pico_register_file::register_name(rs());
		break;

	case MVB2R: 
		buf << opcode_string() << ' '
			<< pico_register_file::register_name(rd()) << ", "
			<< pico_register_file::register_name(rs()+BB_REG);
		break;

	case MVT2S: 
		buf << opcode_string() << ' '
			<< pico_register_file::register_name(rd()+SP_REG) << ", "
			<< pico_register_file::register_name(rs()+TB_REG);
		break;

	case MVB2S:
		buf << opcode_string() << ' '
			<< pico_register_file::register_name(rd()+SP_REG) << ", "
			<< pico_register_file::register_name(rs()+BB_REG);
		break;

	 case PFX:
		buf << opcode_string() << ' '
			<< to_string(signed_immediate());
			break;

	case PUSHS: case POPS: case SSP2REG: case REG2SSP:
	case JR:
		buf << opcode_string() << ' '
		    << pico_register_file::register_name(rd());
			break;

	default:
		buf << opcode_string();
		break;
	}

	return string(buf.str(), buf.pcount());
}

void pico_instruction::set(address_type a, data_type b)
{
	static const opcode_type opcode_table[] = {
	  R_TYPE_OP, HALT,     ANDI,     ORI, 
	  XORI,      RESERVED, ADDI,     SUBI, 
	  JALR,      BNEZ,     BEQZ,     BMI, 
	  BPL,       JAL,      JR,       JMP, 
	  CALL,		 RTN,      RESERVED, RESERVED,
	  MULI,      MULFII,   MULFFI,   RESERVED, 
	  RESERVED,  RESERVED, PFX,		 CMPI,
	  LDLI,      LDHI,     RESERVED, S_TYPE_OP,
	};

	static const opcode_type r_type_table[] = {
	  NOP,       MV,       AND,      OR,
	  XOR,       NOT,      ADD,      SUB,
	  LD,        ST,       LB,       SB,
	  SL,        SR,       CMP,		 RESERVED,
	  RFI,       EINT,     LIAR,     SIAR,     
	  DINT,      MUL,      MULFI,    MULFF,
	  };

	static const opcode_type s_type_table[] = {
	  PUSH,      POP,      LOOK,     NEXT,
	  PUSHS,     POPS,     SSP2REG,  REG2SSP,
	  RESERVED,  RESERVED, RESERVED, RESERVED,
	  RESERVED,  RESERVED, RESERVED, RESERVED,
	  MVR2S,     MVS2R,    MVR2T,    MVT2R,
	  MVR2B,     MVB2R,    MVT2S,    MVB2S,
	};
	pc_ = a; image_ = b;
	opcode_ = opcode_table[int(image() >> 11)];
	switch(opcode_){
	case MULI:
		operation_ = I_TYPE;
		opcode_ = MULI;
		break;

	case MULFII:
		operation_ = I_TYPE;
		opcode_ = MULFII;
		break;
	case MULFFI:
		operation_ = I_TYPE;
		opcode_ = MULFFI;
		break;
	case CALL:
		operation_ = J_TYPE;
		opcode_ = CALL;
		break;
	case RTN:
		operation_ = J_TYPE;
		opcode_ = RTN;
		break;
	case BEQZ: 
		operation_ = J_TYPE;
		opcode_ = BEQZ;
		break;
	case BNEZ:
		operation_ = J_TYPE;
		opcode_ = BNEZ;
		break;
	case ADDI: 
		operation_ = I_TYPE;
		opcode_ = ADDI;
		break;
	case SUBI:
		operation_ = I_TYPE;
		opcode_ = SUBI;
		break;
	case ANDI:
		operation_ = I_TYPE;
		opcode_ = ANDI;
		break;
	case XORI:
		operation_ = I_TYPE;
		opcode_ = XORI;
		break;
	case ORI:
		operation_ = I_TYPE;
		opcode_ = ORI;
		break;
	case LDLI:
		operation_ = I_TYPE;
		opcode_ = LDLI;
		break;
		/*
	case LDHI:
		operation_ = I_TYPE;
		opcode_ = LDHI;
		break;
		*/
	case CMPI:
		operation_ = I_TYPE;
		opcode_ = CMPI;
		break;
	case JMP:
		operation_ = J_TYPE;
		opcode_ = JMP;
		break;
	case JR:
		operation_ = J_TYPE;
		opcode_ = JR;
		break;
		/*
	case JAL:
		operation_ = J_TYPE;
		opcode_ = JAL;
		break;
		*/
	case BMI:
		operation_ = J_TYPE;
		opcode_ = BMI;
		break;
	case BPL:
		operation_ = J_TYPE;
		opcode_ = BPL;
		break;
	case PFX:
		operation_ = I_TYPE;
		opcode_ = PFX;
		break;
	case HALT:
		operation_ = R_TYPE;
		opcode_ = HALT;
		break;
	case R_TYPE_OP:
		operation_ = R_TYPE;
		opcode_ = r_type_table[int(image() & 0x001f)];
		break;
	case S_TYPE_OP:
		operation_ = S_TYPE;
		opcode_ = s_type_table[int(image() & 0x001f)];
		break;
	default:
		break;
	}
}

void pico_instruction::output(ostream &os) const
{
	os << operation_string();
}

ostream& operator<<(ostream& os, const pico_instruction& a)
{
	if(os){
		a.output(os);
	}

	return os;
}

