/*
 * <<< pico_integer_unit.cc >>>
 *
 * --- pico integer unit class 'pico_integer_unit'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cassert>
#include <cstring> // needed only for SGI C++ compiler
#include <iomanip> // needed only for SGI C++ compiler
#include <iostream>
#include "pico_integer_unit.h"
#include "receive_reg.h"
#include "pico_memory_map.h"

//#define IF_DEBUG
//#define UP_SEND_D
//#define DEBUG

using namespace std;

pico_integer_unit::pico_integer_unit(mapped_memory<word>& c, vision_control_unit<word>& e, vision_control_unit<word>& f, pico_timer_unit& g)
	: local_memory(c),
	  cont_unit0(e),
	  cont_unit1(f),
	  ti_unit(g),
	  mult_delay(false),
	  mult_delay2(false),
	  mult_delay3(false),
	  delay_slot_(false),
	  rtn_delay_slot_(false),
	  rtn_delay_slot2_(false),
	  memory_address_(0),
	  memory_access_mode_(0)
{}

pico_integer_unit::pico_integer_unit(const pico_integer_unit& a,
	mapped_memory<word>& d, vision_control_unit<word>& e, vision_control_unit<word>& f, pico_timer_unit& g)
	: inst_buf(a.inst_buf),
	  gpr(a.gpr),
	  mul_unit(a.mul_unit),
	  local_memory(d),
	  cont_unit0(e),
	  cont_unit1(f),
	  ti_unit(g),
	  pc_(a.pc_),
	  mult_delay(false),
	  mult_delay2(false),
	  mult_delay3(false),
	  delay_slot_(false),
	  rtn_delay_slot_(false),
	  rtn_delay_slot2_(false),
	  memory_address_(0),
	  memory_access_mode_(0)
{}

pico_integer_unit::~pico_integer_unit()
{

}

pico_integer_unit& pico_integer_unit::operator=(const pico_integer_unit& a)
{
	if (this != &a) {
		inst_buf = a.inst_buf;
		gpr = a.gpr;
		mul_unit = a.mul_unit;
	}
	return *this;
}

inline void pico_integer_unit::instruction_fetch(void)
{
	const address_type adr = pc();
		inst_buf.if_stage().instruction_fetch(adr,
			local_memory.read(adr));

#ifdef IF_DEBUG
		cout << "IF : " << setw(8) << hex << adr << "  "; 
		cout << hex << local_memory.read(adr) << endl;
#endif //IF_DEBUG
}

inline void pico_integer_unit::register_fetch(void)
{
	pico_pipeline_stage& rf_stage(inst_buf.rf_stage());
	rf_stage.register_fetch(gpr);

	switch(rf_stage.opcode()){
	
	case pico_instruction::MUL:
	case pico_instruction::MULI:
	case pico_instruction::MULFI:
	case pico_instruction::MULFF:
	case pico_instruction::MULFII:
	case pico_instruction::MULFFI:
		mult_delay = true;
		break;
	//case pico_instruction::RTN:
		//rtn_delay_slot2_ = true;
		//mult_delay = true;
	//	break;
	default:
		break;
	}

	//if(delay_slot2_ != true && mult_delay != true){
	if(delay_slot2_ != true){
		rf_stage.forwarding(inst_buf.ex_stage());

	}
}

inline void pico_integer_unit::execute(void)
{
	pico_pipeline_stage& ex_stage(inst_buf.ex_stage());
	ex_stage.execute(gpr);
	memory_access_mode_ = 0;
	switch (ex_stage.opcode()) {
	//case pico_instruction::JAL:
	case pico_instruction::JMP:
	case pico_instruction::JR:
	case pico_instruction::BNEZ:
	case pico_instruction::BEQZ:
	case pico_instruction::BMI:
	case pico_instruction::BPL:
		delay_slot_ = ex_stage.is_branch();
		pc() = ex_stage.destination_pc();
		break;
	case pico_instruction::MUL:
		mul_unit.mult(ex_stage.source1(), ex_stage.source2());
		ex_stage.mul_store(lo(), hi());
		break;
	case pico_instruction::MULI:
		mul_unit.mult(ex_stage.source1(), ex_stage.source2());
		ex_stage.mul_store(lo(), hi());
		break;
	case pico_instruction::MULFI:
		mul_unit.mult(ex_stage.source1(), ex_stage.source2());
		ex_stage.mul_store(lo(), hi());
		break;
	case pico_instruction::MULFF:
		mul_unit.mulff(ex_stage.source1(), ex_stage.source2());
		ex_stage.mul_store(lo(), hi());
		break;
	case pico_instruction::MULFFI:
		mul_unit.mulff(ex_stage.source1(), ex_stage.source2());
		ex_stage.mul_store(lo(), hi());
		break;
	case pico_instruction::MULFII:
		mul_unit.mult(ex_stage.source1(), ex_stage.source2());
		ex_stage.mul_store(lo(), hi());
		break;
		/*
	case pico_instruction::DIV:
		mul_unit.div(ex_stage.source1(), ex_stage.source2());
		break;
		*/
	case pico_instruction::LD:
		// load
	{
		const address_type adr = ex_stage.word_aligned_access_address();
		memory_address_ = adr;
		memory_access_mode_ = 1;
		data_type d;
		switch(adr){
		case LEFTD0: 
			d = cont_unit0.right_r_d_reg();
			break;
		case LEFTC0:
			d = cont_unit0.right_r_s_reg();
			cont_unit0.address_buffer() = RIGHT_RECEIVE_STATUS; 
			break;
		case RIGHTD0:
			d = cont_unit0.left_r_d_reg();
			break;
		case RIGHTC0:
			d = cont_unit0.left_r_s_reg();
			cont_unit0.address_buffer() = LEFT_RECEIVE_STATUS; 
			break;
		case UPD0:
			d = cont_unit0.down_r_d_reg();
			break;
		case UPC0:
			d = cont_unit0.down_r_s_reg();
			cont_unit0.address_buffer() = DOWN_RECEIVE_STATUS; 
			break;
		case DOWND0:
			d = cont_unit0.up_r_d_reg();
			break;
		case DOWNC0:
			d = cont_unit0.up_r_s_reg();
			cont_unit0.address_buffer() = UP_RECEIVE_STATUS; 
			break;
		case INPUT0:
			d = cont_unit0.input_data();
			cont_unit0.input_decode0();
			break;
		case INPUT_C0:
			cont_unit0.input_decode1();
			break;
		case STATUS0:
			d = cont_unit0.get_transmit_status();
			break;
		case LEFTD1: 
			d = cont_unit1.right_r_d_reg();
			break;
		case LEFTC1:
			d = cont_unit1.right_r_s_reg();
			cont_unit1.address_buffer() = RIGHT_RECEIVE_STATUS; 
			break;
		case RIGHTD1:
			d = cont_unit1.left_r_d_reg();
			break;
		case RIGHTC1:
			d = cont_unit1.left_r_s_reg();
			cont_unit1.address_buffer() = LEFT_RECEIVE_STATUS; 
			break;
		case UPD1:
#ifdef UP_SEND_D
			cout << "UPD1 Access\n" << endl;
#endif			
			d = cont_unit1.down_r_d_reg();
			break;
		case UPC1:
			d = cont_unit1.down_r_s_reg();
			cont_unit1.address_buffer() = DOWN_RECEIVE_STATUS; 
			break;
		case DOWND1:
			d = cont_unit1.up_r_d_reg();
			break;
		case DOWNC1:
			d = cont_unit1.up_r_s_reg();
			cont_unit1.address_buffer() = UP_RECEIVE_STATUS; 
			break;
		case INPUT1:
			d = cont_unit1.input_data();
			cont_unit1.input_decode0();
			break;
		case INPUT_C1:
			cont_unit1.input_decode1();
			break;
		case STATUS1:
			d = cont_unit1.get_transmit_status();
			break;
			// Timer Set 
			/*
		case TIMER_COUNT:
			break;
		case TIMER_OP:
			break;
		case TIMER_VALUE:
			break;	
			*/
		default:
			d = local_memory.read(adr);
			break;
		}
			ex_stage.load_fetch_for_big_endian(d);
			break;
	}
	case pico_instruction::POP:
	case pico_instruction::LOOK:
	case pico_instruction::POPS:
		// load
	{
		const address_type adr = ex_stage.word_aligned_access_address();
				ex_stage.load_fetch_for_big_endian(local_memory.read(adr));
	}
		break;
	case pico_instruction::RTN:
		// load
	{
		const address_type adr = ex_stage.word_aligned_access_address();
				ex_stage.load_fetch_for_big_endian(local_memory.read(adr));
			pc() = ex_stage.destination_pc();
			delay_slot_ = true;
			rtn_delay_slot2_ = true;
	}
		break;
	case pico_instruction::ST: 
		//store
	{
		const address_type adr = ex_stage.access_address();
		memory_address_ = adr;
		memory_access_mode_ = 2;
		switch(adr){
		case LEFTD0:
		case LEFTC0:
		case RIGHTD0:
		case RIGHTC0:
		case UPD0:
		case UPC0:
		case DOWND0:
		case DOWNC0:
		case RESULT0:
		case RESULT_C0:
		case STATUS0:
			cont_unit0.address_buffer() = adr;
			cont_unit0.data_buffer() = ex_stage.access_data();
			break;
		case LEFTD1:
		case LEFTC1:
		case RIGHTD1:
		case RIGHTC1:
		case UPD1:
		case UPC1:
		case DOWND1:
		case DOWNC1:
		case RESULT1:
		case RESULT_C1:
		case STATUS1:
			cont_unit1.address_buffer() = (adr - 0x40);
			cont_unit1.data_buffer() = ex_stage.access_data();
			break;
			/*
		case TIMER_COUNT:
			ti_unit.timer_clock_count() = ex_stage.access_data();
			break;
		case TIMER_OP:
			ti_unit.set_operation() = ex_stage.access_data();
			break;
		case TIMER_VALUE:
			break;	
			*/
		default:
			local_memory.write(adr, ex_stage.access_data()); 
					
			break;
		}
			//local_memory.write(adr, ex_stage.access_data()); 
		break;
		}
	case pico_instruction::PUSH:
	case pico_instruction::PUSHS:
		local_memory.write(ex_stage.access_address(), 
					ex_stage.access_data()); 
		break;
	case pico_instruction::CALL:
		local_memory.write(ex_stage.access_address(), 
					ex_stage.access_data()); 
		pc() = ex_stage.destination_pc();
		delay_slot_ = true;
		break;
	case pico_instruction::SSP2REG:
		delay_slot_ = true;
		break;
	default:
		break;
	}


}

inline void pico_integer_unit::writeback(void)
{
	inst_buf.wb_stage().writeback(gpr);
}

void pico_integer_unit::clock(void)
{
	// timer

	//ti_unit -> clock_in();
	// mult_delay mult no maeno meirei no writeback ha zikkou
	// delay_slot branch  no delay
	// rtn_delay_slot_  return de forwarding no seigyo
	// rtn
	// mult_delay2 = mult no delay
	// run cycle

	if(rtn_delay_slot2_ == true){
		rtn_delay_slot2_ = false;
	}
	else{
		mul_unit.clock();

		if(rtn_delay_slot_ == true){
			rtn_delay_slot_ = false;
			delay_slot2_ = true;
		}
		else{

			if(delay_slot2_ == true){
				delay_slot2_ = false;
			}
			else{
				writeback();
			}

			execute();
		}	

		if(delay_slot_ == true){
			rtn_delay_slot_ = true;
			delay_slot_ = false;

			instruction_fetch();
			pc() += sizeof_data_type;
			print_pc_ = pc();
			inst_buf.step();

		}
		else{

			if(mult_delay == false && mult_delay3 == false){
				register_fetch();
				instruction_fetch();
			}
			else if(mult_delay3 == true){

			}
			else{ //mult_delay == true
				mult_delay = false;
				mult_delay2 = true;
				register_fetch();
				instruction_fetch();
			}

			if(mult_delay2 == false){
				pc() += sizeof_data_type;
			}

			if(mult_delay == false){
				print_pc_ = pc();
			}
			//inst_buf.step();

			if(mult_delay2 == false && mult_delay3 == false){
				inst_buf.step();
			}
			else if(mult_delay3 == true){
				mult_delay3 = false;
			}
			else{ //mult_delay2 == true
				mult_delay2 = false;
				inst_buf.step();
				mult_delay3 = true;
			}
		}
	}
}

void pico_integer_unit::reset(void)
{
	inst_buf.clear();
	gpr.clear();
	mul_unit.clear();
}

void pico_integer_unit::output(ostream& os) const
{
    ostream::fmtflags flags = os.flags();
	const char fill = os.fill();
	os << hex << setfill('0');
	os << gpr << '\n'
	   <<  "pc:" << setw(8) << pc();
	if (mul_unit.ready()) {
		os << " hi:" << setw(8) << hi()
		   << " lo:" << setw(8) << lo();
	} else {
		os << " hi:-locked- lo:-locked-";
	}
	os << '\n'
	   << "rf: " << inst_buf.rf_stage() << '\n'
	   << "ex: " << inst_buf.ex_stage() << '\n'
	   << "wb: " << inst_buf.wb_stage();
	os.flags(flags);
	os.fill(fill);
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
}

bool pico_integer_unit::output(ostream& os, const string& s) const
{
	if (s == "instruction_buffer" || s == "ibuf") {
		os << inst_buf;
		return true;
	} else if (s == "gpr") {
		os << gpr;
		return true;
	} else if (s == "stall") {
		//os << stall_buf;
		return true;
	} else if (s == "pc") {
		ostream::fmtflags flags = os.flags();
		os << "0x" << hex << pc_;
		os.flags(flags);
		return true;
	}
	return false;
}

string pico_integer_unit::opcode_string(int stage) const
{
	switch(stage){
	case 1:
		return inst_buf.if_stage().operation_string();
		break;
	case 2:
		return inst_buf.rf_stage().operation_string();
		break;
	case 3:
		return inst_buf.ex_stage().operation_string();
		break;
	case 4:
		return inst_buf.wb_stage().operation_string();
		break;
	default:
		return "";
	}
}

/*
void pico_integer_unit::set_timer_address(address_type a)
{
	ti_unit -> timer_memory_address() = a;
}

void pico_integer_unit::set_timer_count(data_type d)
{
	ti_unit -> timer_clock_count() = d;
}
*/

ostream& operator<<(ostream& os, const pico_integer_unit& a)
{
	if (os) a.output(os);
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
	return os;
}
