/*
 * <<< pico_pipeline_stage.cc >>>
 *
 * --- pico pipeline state class 'pico_pipeline_stage'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstring> // needed only for SGI C++ compiler
#include <iomanip> // needed only for SGI C++ compiler
#include <iostream>
#include <strstream>
#include "pico_pipeline_stage.h"

//#define DEBUG

#define SP_REG 9
#define TB_REG 17
#define BB_REG 25

using namespace std;

inline pico_pipeline_stage::data_type expand_signed
	(pico_pipeline_stage::data_type x)
{
	return (x & 0x8000UL) ? (x | 0xffff0000UL) : x;
}

template <class T>
static string to_string(T x)
{
	strstream buf;
	T tmp;
	if(x > 0 || x == 0){
		tmp = x;
	}
	else {
		buf << '-';
		tmp = x;
	}

	if(tmp == 0 || ((tmp & 0xff) != 0 && ((tmp % 1000) == 0 ||
		tmp < 1000))){

		buf << tmp;
	}
	else {
		buf << hex << "0x" << tmp;
	}

	string str;
	buf >> str;

	return str;
}

pico_pipeline_stage::register_set::register_set(void)
{

}

pico_pipeline_stage::register_set::register_set
	(pico_pipeline_stage::data_type a, int b)
	: value(a), number(b)
{

}

inline void pico_pipeline_stage::register_set::clear(void)
{
	value = 0, number = 0;
}

pico_pipeline_stage::pico_pipeline_stage(void)
{
	clear();
}

pico_pipeline_stage::pico_pipeline_stage(const pico_pipeline_stage &a)
	: buf(a.buf), src1(a.src1), src2(a.src2), dst(a.dst),
	  reg_in(a.reg_in), io_flags(a.io_flags), acc_adr(a.acc_adr),
	  acc_dat(a.acc_dat), dst_pc(a.dst_pc) 
{

}

pico_pipeline_stage::~pico_pipeline_stage(void)
{

}

void pico_pipeline_stage::instruction_fetch
	(pico_pipeline_stage::address_type a, pico_pipeline_stage::data_type b)
{
	const unsigned short JAL_STACK = (0x07 & 0x07);
	buf.set(a, b);
	switch(opcode()){
	/*
	case pico_instruction::NOP:
	case pico_instruction::HALT:
		reg_in = 0;
		io_flags = 0;
		break;
		*/
	case pico_instruction::ADD:
	case pico_instruction::SUB:
	case pico_instruction::MV:
	case pico_instruction::MUL:
	case pico_instruction::MULFI:
	case pico_instruction::MULFF:
		reg_in = 2;
		io_flags = dst_out;
		src1.number = buf.rd();
		src2.number = buf.rs();
		dst.number = buf.rd();
		break;

	case pico_instruction::CMP:
		reg_in = 2;
		io_flags = 0;
		src1.number = buf.rd();
		src2.number = buf.rs();
		break;
	case pico_instruction::ADDI:
	case pico_instruction::SUBI:
	case pico_instruction::MULI:
	case pico_instruction::MULFII:
	case pico_instruction::MULFFI:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.rd();
		dst.number = buf.rd();
		break;

	case pico_instruction::CMPI:
		reg_in = 1;
		io_flags = 0;
		src1.number = buf.rd();
		break;
	case pico_instruction::AND:
	case pico_instruction::OR:
	case pico_instruction::XOR:
		reg_in = 2;
		io_flags = dst_out;
		src1.number = buf.rd();
		src2.number = buf.rs();
		dst.number = buf.rd();
		break;

	case pico_instruction::ANDI:
	case pico_instruction::ORI:
	case pico_instruction::XORI:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.rd();
		dst.number = buf.rd();
		break;

	case pico_instruction::SL:
	case pico_instruction::SR:
		reg_in = 2;
		io_flags = dst_out;
		src1.number = buf.rd();
		src2.number = buf.rs();
		dst.number = buf.rd();
		break;

	case pico_instruction::LD:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.rs();
		dst.number = buf.rd();
		break;
	case pico_instruction::LDLI:
//	case pico_instruction::LDHI:
		reg_in = 0;
		io_flags = dst_out;
		dst.number = buf.rd();
		break;
	case pico_instruction::PFX:
		reg_in = 0;
		io_flags = dst_out;
		dst.number = PREFIX_REGISTER;
		break;

	case pico_instruction::ST:
		reg_in = 2;
		io_flags = 0;
		src1.number = buf.rd();
		src2.number = buf.rs();
		break;
	case pico_instruction::JMP:
	case pico_instruction::BPL:
	case pico_instruction::BMI:
	case pico_instruction::BEQZ:
	case pico_instruction::BNEZ:
		reg_in = 0;
		io_flags = 0;
		break;
	case pico_instruction::JAL:
		reg_in = 0;
		io_flags = dst_out;
		dst.number = JAL_STACK;
		break;
	case pico_instruction::CALL:
	case pico_instruction::RTN:
		reg_in = 0;
		io_flags = 0;
		break;
	case pico_instruction::PUSHS:
		reg_in = 1;
		io_flags = 0;
		src1.number = buf.rd();
		break;
	case pico_instruction::POPS:
		reg_in = 0;
		io_flags = dst_out;
		dst.number = buf.rd();
		break;
	case pico_instruction::SSP2REG:
		reg_in = 0;
		io_flags = dst_out;
		dst.number = buf.rd();
		break;
	case pico_instruction::REG2SSP:
		reg_in = 1;
		io_flags = 0;
		src1.number = buf.rd();
		break;
	case pico_instruction::JR:
		reg_in = 1;
		io_flags = 0;
		src1.number = buf.rd();
		break;
	case pico_instruction::PUSH:
		reg_in = 2;
		io_flags = 0;
		src1.number = (buf.rd() + SP_REG);
		src2.number = buf.rs();
		break;
	case pico_instruction::POP:
	case pico_instruction::NEXT:
	case pico_instruction::LOOK:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = (buf.rs() + SP_REG);
		dst.number = buf.rd();
		break;
	case pico_instruction::MVR2S:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.rs();
		dst.number = (buf.rd() + SP_REG);
		break;
	case pico_instruction::MVS2R:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = (buf.rs() + SP_REG);
		dst.number = buf.rd();
		break;
	case pico_instruction::MVR2T:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.rs();
		dst.number = (buf.rd() + TB_REG);
		break;

	case pico_instruction::MVT2R:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = (buf.rs() + TB_REG);
		dst.number = buf.rd();
		break;
	case pico_instruction::MVR2B:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.rs();
		dst.number = (buf.rd() + BB_REG);
		break;
	case pico_instruction::MVB2R:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = (buf.rs() + BB_REG);
		dst.number = buf.rd();
		break;
	case pico_instruction::MVT2S:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = (buf.rs() + TB_REG);
		dst.number = (buf.rd() + SP_REG);
		break;

	case pico_instruction::MVB2S:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = (buf.rs() + BB_REG);
		dst.number = (buf.rd() + SP_REG);
		break;	
	default:
		reg_in = 0;
		io_flags = 0;
		break;
	}

}

void pico_pipeline_stage::register_fetch(const pico_register_file & rf)
{

	switch(reg_in){
	case 1:
		src1.value = rf[source1_number()];
		break;
	case 2:
		src1.value = rf[source1_number()];
		src2.value = rf[source2_number()];
		break;
	default:
		break;
	}


}

void pico_pipeline_stage::execute(pico_register_file &rf)
{
	if(is_i_type()){
		src2.value = buf.immediate() | rf[PREFIX_REGISTER];

	}

	change_flag = false;

	switch(buf.opcode()){

	case pico_instruction::NOP:
	case pico_instruction::HALT:
		break;

	case pico_instruction::ADD:
	case pico_instruction::ADDI:
		dst.value = source1() + source2();
		change_flag = true;
		break;

	case pico_instruction::SUB:
	case pico_instruction::SUBI:
		dst.value = source1() - source2();
		change_flag = true;
		break;

	case pico_instruction::AND:
	case pico_instruction::ANDI:
		dst.value = source1() & source2();
		change_flag = true;
		break;

	case pico_instruction::OR:
	case pico_instruction::ORI:
		dst.value = source1() | source2();
		change_flag = true;
		break;
	
	case pico_instruction::XOR:
	case pico_instruction::XORI:
		dst.value = source1() ^ source2();
		change_flag = true;
		break;

	case pico_instruction::SL:
		dst.value = source2() << 1;
		change_flag = true;
		break;

	case pico_instruction::SR:
		dst.value = source2() >> 1;
		change_flag = true;
		break;

	case pico_instruction::LD:
		acc_adr = source1();
		change_flag = true;
		break;
		
	case pico_instruction::LDLI:
		dst.value = source2();
		//dst.value = buf.immediate();
		break;

/*
	case pico_instruction::LDHI:
		dst.value = buf.immediate() << 8;
		break;
		*/
	case pico_instruction::PFX:
		dst.value = buf.immediate() << 8;
		break;

	case pico_instruction::ST:
		acc_adr = source1();
		acc_dat = source2();
		break;
	case pico_instruction::MV:
		dst.value = source2();
		change_flag = true;
		break;
	
	case pico_instruction::CALL:
		rf[SYSTEM_STACK] -= 0x02;
		acc_adr = rf[SYSTEM_STACK];
		acc_dat = buf.pc() + 0x02;
		
		if(image() & 0x0400){
			unsigned short hoge;
			hoge = (~((image() - 0x0001) & 0x07ff) & 0x7ff);
			dst_pc = buf.pc() + sizeof_data_type  - (hoge << 1); 
		}
		else{
			dst_pc = buf.pc() + sizeof_data_type + 
					((image() & 0x07ff) << 1);
		}
		branch_flag_ = true;
		break;
	case pico_instruction::RTN:
		acc_adr = rf[SYSTEM_STACK];
		rf[SYSTEM_STACK] += 0x02;
		branch_flag_ = true;
		break;
	case pico_instruction::PUSHS:
		rf[SYSTEM_STACK] -= 0x02;
		acc_adr = rf[SYSTEM_STACK];
		acc_dat = source1();
		break;
	case pico_instruction::POPS:
		acc_adr = rf[SYSTEM_STACK];
		rf[SYSTEM_STACK] += 0x02;
		break;
	case pico_instruction::SSP2REG:
		dst.value = rf[SYSTEM_STACK];
		break;
	case pico_instruction::REG2SSP:
		rf[SYSTEM_STACK] = source1();
		break;
	case pico_instruction::JR:
		dst_pc = source1();
		branch_flag_ = true;
		break;
	
	/*
	case pico_instruction::JAL:
		dst.value = buf.pc() + 0x02;
		
		if(image() & 0x0400){
			unsigned short hoge;
			hoge = (~((image() - 0x0001) & 0x07ff) & 0x7ff);
			dst_pc = buf.pc() + sizeof_data_type  - (hoge << 1); 
		}
		else{
			dst_pc = buf.pc() + sizeof_data_type + 
					((image() & 0x07ff) << 1);
		}
		branch_flag_ = true;
		break;
		*/

	case pico_instruction::JMP:
		if(image() & 0x0400){
			unsigned short hoge;
			hoge = (~((image() - 0x0001) & 0x07ff) & 0x7ff);
			dst_pc = buf.pc() + sizeof_data_type  - (hoge << 1) ; 
		}
		else{
			dst_pc = buf.pc() + sizeof_data_type +  
					((image() & 0x07ff) << 1);
		}
		branch_flag_ = true;

		break;
	case pico_instruction::BMI:
		if(rf[33] & NEGATIVE_FLAG){
			if(image() & 0x0400){
				unsigned short hoge;
				hoge = (~((image() - 0x0001) & 0x07ff) & 0x7ff);
				dst_pc = buf.pc() + sizeof_data_type  - (hoge << 1) ; 
			}
			else{
				dst_pc = buf.pc() + sizeof_data_type +  
					((image() & 0x07ff) << 1);
			}
			branch_flag_ = true;
		}
		else{ 
			dst_pc = buf.pc() + sizeof_data_type * 2;
			branch_flag_ = false;
		}
		break;
	case pico_instruction::BPL:
		if((rf[33] != NEGATIVE_FLAG) && (rf[33] != ZERO_FLAG)){
			if(image() & 0x0400){
				unsigned short hoge;
				hoge = (~((image() - 0x0001) & 0x07ff) & 0x7ff);
				dst_pc = buf.pc() + sizeof_data_type  - (hoge << 1) ; 
			}
			else{
				dst_pc = buf.pc() + sizeof_data_type +  
					((image() & 0x07ff) << 1);
			}
			branch_flag_ = true;
		}
		else{
			dst_pc = buf.pc() + sizeof_data_type * 2;
			branch_flag_ = false;
		}
		break;
		
	case pico_instruction::BEQZ:
		if(rf[33] & ZERO_FLAG){
			if(image() & 0x0400){
				unsigned short hoge;
				hoge = (~((image() - 0x0001) & 0x07ff) & 0x7ff);
				dst_pc = buf.pc() + sizeof_data_type  - (hoge << 1) ; 
			}
			else{
				dst_pc = buf.pc() + sizeof_data_type +  
					((image() & 0x07ff) << 1);
			}
			branch_flag_ = true;
		}
		else{
			dst_pc = buf.pc() + sizeof_data_type * 2;
			branch_flag_ = false;
		}
		break;
	case pico_instruction::BNEZ:
		if(!(rf[33] & ZERO_FLAG)){
			if(image() & 0x0400){
				unsigned short hoge;
				hoge = (~((image() - 0x0001) & 0x07ff) & 0x7ff);
				dst_pc = buf.pc() + sizeof_data_type  - (hoge << 1) ; 
			}
			else{
				dst_pc = buf.pc() + sizeof_data_type +  
					((image() & 0x07ff) << 1);
			}
			branch_flag_ = true;
		}
		else{
			dst_pc = buf.pc() + sizeof_data_type * 2;
			branch_flag_ = false;
		}
		break;
	case pico_instruction::PUSH:
		if(source1() > rf[source1_number() - SP_REG + TB_REG]){
			rf[source1_number()] -= 0x02;
			acc_adr = (source1() - 0x02);
			acc_dat = source2();
		}
		break;
	case pico_instruction::POP:
		if(source1() < rf[source1_number() - SP_REG + BB_REG]){
			acc_adr = source1();
			rf[source1_number()] += 0x02;
		}
		break;
	case pico_instruction::LOOK:
		if(source1() < rf[source1_number() - SP_REG + BB_REG]){
			acc_adr = source1();
		}
		break;
	case pico_instruction::NEXT:
		if(source1() < rf[source1_number() - SP_REG + BB_REG]){
			rf[source1_number()] += 0x2;
			dst.value = 0x1;
		}
		else {
			dst.value = 0x0;
	 	}	
		break;
	case pico_instruction::CMP:
	case pico_instruction::CMPI:
	{
		
		//int temp = source1() - source2();
		if(source1() == source2()){
			rf[FLAG_REGISTER] = ZERO_FLAG;
		}
		else if((source1() - source2()) & 0x8000UL){
			rf[FLAG_REGISTER] = NEGATIVE_FLAG;
		}
		else{
			rf[FLAG_REGISTER] = 0;
		}
			
		break;
		
	}
	case pico_instruction::MVR2S:
	case pico_instruction::MVS2R:
	case pico_instruction::MVR2T:
	case pico_instruction::MVT2R:
	case pico_instruction::MVR2B:
	case pico_instruction::MVB2R:
	case pico_instruction::MVT2S:
	case pico_instruction::MVB2S:
		dst.value = source1();
		break;
//	case pico_instruction::MUL:
//	case pico_instruction::MULI:
//	case pico_instruction::MULFI:
//	case pico_instruction::MULFF:
//	case pico_instruction::MULFII:
//	case pico_instruction::MULFFI:
//		change_flag = true;
//		break;
	default:
		break;
	}

	if(change_flag){
		if(dst.value == 0){
			rf[FLAG_REGISTER] = ZERO_FLAG;
		}
		else if(dst.value & 0x8000){
			rf[FLAG_REGISTER] = NEGATIVE_FLAG;
		}
		else{
			rf[FLAG_REGISTER] = 0;
		}
	}

/*
	if(buf.opcode() != pico_instruction::CMP &&
				buf.opcode() != pico_instruction::CMPI){
		//clear flag
		rf[FLAG_REGISTER] = 0;
	}
	*/
	if(buf.opcode() != pico_instruction::PFX){
		rf[PREFIX_REGISTER] = 0;
	}
}

void pico_pipeline_stage::mul_store(pico_pipeline_stage::data_type a,
		pico_pipeline_stage::data_type b)
{
	dst.value = a;
	//dst.value = a + (b << 8) ;
}

void pico_pipeline_stage::load_fetch_for_big_endian
	(pico_pipeline_stage::data_type a)
{
	data_type tmp;
	switch (buf.opcode()) {
	case pico_instruction::LD:
	case pico_instruction::POP:
	case pico_instruction::POPS:
	case pico_instruction::LOOK:
		dst.value = a;
		break;
	case pico_instruction::RTN:
		dst_pc = a;
	default:
		break;
	}
}

void pico_pipeline_stage::load_fetch_for_little_endian
	(pico_pipeline_stage::data_type a)
{
	data_type tmp;
	switch (buf.opcode()) {
	case pico_instruction::LB:
		tmp = (a >> (int(access_address() & 0x3) * 8)) & 0xff;
		dst.value = ((tmp & 0x80) == 0) ? tmp : (0xffffff00UL | tmp);
		break;
	case pico_instruction::LD:
	case pico_instruction::POP:
	case pico_instruction::LOOK:
		dst.value = a;
		break;
	case pico_instruction::RTN:
		dst_pc = a;
	default:
		break;
	}
}

void pico_pipeline_stage::partial_word_store_fetch_for_big_endian
	(pico_pipeline_stage::data_type tmp1)
{
	data_type tmp2 = access_data();
	switch (opcode()) {
	case pico_instruction::ST:
		acc_dat = tmp1;
		break;
	default:
		break;
	}
}

void pico_pipeline_stage::partial_word_store_fetch_for_little_endian
	(pico_pipeline_stage::data_type tmp1)
{
	data_type tmp2 = acc_dat;
	switch (buf.opcode()) {
	case pico_instruction::ST:
			acc_dat = tmp1;
		break;
	default:
		break;
	}
}

void pico_pipeline_stage::writeback(pico_register_file& rf)
{
	if(is_destination_output()){
//		if(destination() > 0){

//		}
//		else{
			rf[destination_number()] = destination();
//		}
	}
}

void pico_pipeline_stage::clear(void)
{
	instruction_fetch(0, 0);
	src1.clear();
	src2.clear();
	dst.clear();
	reg_in = io_flags = 0;
	acc_adr = acc_dat = dst_pc = 0;
}

void pico_pipeline_stage::output(ostream &os) const
{
	os << buf;

}

ostream& operator<<(ostream& os, const pico_pipeline_stage& a)
{
	if(os){
		a.output(os);
	}

	return os;
}
