/*
 * <<< pico_simple_test.cc >>>
 *
 * --- Simple test program for pico
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
#include <isis/memory_control_unit.h>
#include <isis/multi_io_unit.h>
#include <isis/bus_error_detector.h>
#include <isis/halt_detector.h>
#include <isis/bus_port.h>
//#include <isis/r3010.h>
//#include "mips_executable.h"
#include "pico.h"
//#include "pico_directmap_cache.h"
#include <isis/isis.h>
#include "pico_typedef.h"

#define MEM_DUMP
#define MEM_DUMP2
#define REG_DUMP 


static int loader(istream&);
static unsigned int string_hex_to_unsigned_int(const string& a);

int main(int, char** argv)
{
	typedef pico_word word;
	const word local_memory_address = 0;
	const word local_memory_size = LOCAL_MEMORY_SIZE;
//	const word local_memory_size = 0x20000000;
	const word multi_io_area_address = 0x4000;
	//const word multi_io_area_address = 0xbfd0;
//	const word multi_io_area_address = 0xbfd00000;
	const word multi_io_area_size = 0x5000;
	
	halt_detector<word> mon;
	bus_port<word> bus_if;

	mon.port_ref().connect(bus_if);


#ifdef REG_DUMP
	ofstream reg_dump("reg_dump", ios::out | ios::trunc);
	ofstream sp_dump("sp_dump", ios::out | ios::trunc);
	ofstream tb_dump("tb_dump", ios::out | ios::trunc);
	ofstream bb_dump("bb_dump", ios::out | ios::trunc);
#endif //REG_DUMP

	// read commandline argument for simulator
	if (*argv == NULL) {
		cerr << "No executable file specified." << endl;
		return 1;
	}

	// create units and connect each other
	pico pu;
	memory_control_unit<word> mem_ctl;
	//mapped_memory<word> local_mem;
	multi_io_unit<word> multi_io;
	bus_error_detector<word> buserror_mon;
	pu.port_ref().connect(mem_ctl.port_ref());
	pu.port_ref().connect(multi_io.port_ref());
	pu.port_ref().connect(buserror_mon.port_ref());
	//mem_ctl.connect_memory(*pu.local_memory);

	// setup memory map
	//local_mem.set_top(local_memory_address);
	//local_mem.resize(local_memory_size);
	multi_io.set_address(multi_io_area_address, multi_io_area_size);
	multi_io.set_sysinfo("local_memory_address", local_memory_address);
	multi_io.set_sysinfo("local_memory_size", local_memory_size);

	// setup file table
	multi_io.set_file_table_size(16);
	multi_io.set_file_input_stream(0, cin);
	multi_io.set_file_output_stream(1, cout);
	multi_io.set_file_output_stream(2, cerr);

	// load program to memory
	{
		ifstream file(argv[1], ios::in | ios::nocreate);
		if (file.bad()) return 1;

#ifdef MEM_DUMP
		ofstream mem_dump("mem_dump", ios::out | ios::trunc);
#endif //MEM_DUMP


	int i, j;
	typedef pico_word word_type;
	const size_t buf_size = 1024;
	const int WRITE_SIZE = 0x0024;
	int halt_adr = WRITE_SIZE;
	const size_t sizeof_word_type = 2;
	word_type buf[buf_size];
	word_type adr = 00000000;

	for(i = 0; ; i += sizeof_word_type){
		buf[i / sizeof_word_type] = loader(file);
		if(buf[i / sizeof_word_type] == 2048){
			halt_adr = i + 4;
//			cout << "Set End of Program! " <<  halt_adr <<  endl;
		}
		if(buf[i / sizeof_word_type] == 65336){
			break;
		}
#ifdef DEBUG
		cout << "Read " << buf[i] << endl;
#endif // DEBUG

	}

	// SET HALT ADDRESS
	mon.set_address(halt_adr);

	for(j = 0; j <= i; j += sizeof_word_type){
		pu.local_memory -> write(adr + j, buf[j / sizeof_word_type]);
	}
#ifdef DEBUG
	word_type dat;
	for(j = 0; j <= i; j += sizeof_word_type){
		dat = pu.local_memory -> read(adr + j);
		cout << setw(8) << adr + i << " " << setw(8) << dat << endl;
	}
#endif // DEBUG
#ifdef MEM_DUMP
		pu.local_memory -> dump(mem_dump);
#endif// MEM_DUMP
		pu.program_counter() = 0;
	}

	multi_io.set_commandline_argument((const char* const*)(argv + 1));

	int cl = 0;

	cout << hex << setfill('0');

	int loop = 0;
//	const int WRITE_SIZE = 0x024;

	//int i;
	//cin >> i;


	// execute simulation
	//while (!buserror_mon.is_bus_error() && loop < WRITE_SIZE) {
	while(!buserror_mon.is_bus_error() && !multi_io.is_halt()){
		bus_if.send_single_write_request(pu.program_counter());
		pu.clock_in();
		//mem_ctl.clock_in();
		//multi_io.clock_in();
		//buserror_mon.clock_in();
		mon.clock_in();
		mon.clock_out();
		pu.clock_out();
		//mem_ctl.clock_out();
		//multi_io.clock_out();
		//buserror_mon.clock_out();
		//loop += 2;
#ifdef REG_DUMP
		const int REG_SIZE = 8;
		const int SP_REG = 9;
		const int TB_REG = 17;
		const int BB_REG = 25;
		int i;

		reg_dump << setw(8) << cl << ": ";
		cl++;
		for(i = 0; i < REG_SIZE; i++){
			reg_dump << setw(5) << pu.register_file(i) << " ";
		}
		reg_dump << endl;

		sp_dump << setw(8) << cl << ": ";
		for(i = 0; i < REG_SIZE; i++){
			sp_dump << setw(5) << pu.register_file(i + SP_REG) << " ";
		}
		sp_dump << endl;

		tb_dump << setw(8) << cl << ": ";
		for(i = 0; i < REG_SIZE; i++){
			tb_dump << setw(5) << pu.register_file(i + TB_REG) << " ";
		}
		tb_dump << endl;

		bb_dump << setw(8) << cl << ": ";
		for(i = 0; i < REG_SIZE; i++){
			bb_dump << setw(5) << pu.register_file(i + BB_REG) << " ";
		}
		bb_dump << endl;

		//cout << pu.int_unit.gpr.rf << endl;
#endif //REG_DUMP

		if(mon.is_halt()){
			break;
		}

	}
#ifdef MEM_DUMP2
		ofstream mem_dump2("mem_dump2", ios::out | ios::trunc);
		pu.local_memory -> dump(mem_dump2);
#endif// MEM_DUMP2
	return 0;
}

static int loader(istream& file)
{
	while (1) {
		char c;

		if (!file){
			break;
		}

		if (!(file >> c)){
			return (65336);
			//break;
		}

		if (c == '@') {
			unsigned int adr, c0, c1;
			string adr_str, c0_str, c1_str, tmp;
			file >> adr_str >> c0_str >> c1_str;
			adr = string_hex_to_unsigned_int(adr_str);
			c0 = string_hex_to_unsigned_int(c0_str);
		//	c1 = string_hex_to_unsigned_int(c1_str);

			// check
#ifdef DEBUG
			cout << "adr:" << adr
			 << " (" << c0 << ',' << c1 << ')' << endl;
#endif //DEBUG

//			return (c0 << 8 + c1);
//			return (c0 << 8) + c1;
			return c0;
		}
		while (1) {
			if (!file || !file.get(c)) break;
			if (c == '\n') break;
		}
	}

	return 0;
}
unsigned int string_hex_to_unsigned_int(const string& a)
{
	unsigned int value = 0;
	for (size_t i = 0; i < a.size(); i++) {
		const char c = a[i];
		int t;
		switch (c) {
		case '0': t =  0; break;
		case '1': t =  1; break;
		case '2': t =  2; break;
		case '3': t =  3; break;
		case '4': t =  4; break;
		case '5': t =  5; break;
		case '6': t =  6; break;
		case '7': t =  7; break;
		case '8': t =  8; break;
		case '9': t =  9; break;
		case 'a': t = 10; break;
		case 'A': t = 10; break;
		case 'b': t = 11; break;
		case 'B': t = 11; break;
		case 'c': t = 12; break;
		case 'C': t = 12; break;
		case 'd': t = 13; break;
		case 'D': t = 13; break;
		case 'e': t = 14; break;
		case 'E': t = 14; break;
		case 'f': t = 15; break;
		case 'F': t = 15; break;
		default:  t =  0; break;
		}
		value = (value << 4) + t;
	}
	return value;
}
