/* options.h - options package interfaces */

/* SimpleScalar(TM) Tool Suite
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 * All Rights Reserved. 
 * 
 * THIS IS A LEGAL DOCUMENT, BY USING SIMPLESCALAR,
 * YOU ARE AGREEING TO THESE TERMS AND CONDITIONS.
 * 
 * No portion of this work may be used by any commercial entity, or for any
 * commercial purpose, without the prior, written permission of SimpleScalar,
 * LLC (info@simplescalar.com). Nonprofit and noncommercial use is permitted
 * as described below.
 * 
 * 1. SimpleScalar is provided AS IS, with no warranty of any kind, express
 * or implied. The user of the program accepts full responsibility for the
 * application of the program and the use of any results.
 * 
 * 2. Nonprofit and noncommercial use is encouraged. SimpleScalar may be
 * downloaded, compiled, executed, copied, and modified solely for nonprofit,
 * educational, noncommercial research, and noncommercial scholarship
 * purposes provided that this notice in its entirety accompanies all copies.
 * Copies of the modified software can be delivered to persons who use it
 * solely for nonprofit, educational, noncommercial research, and
 * noncommercial scholarship purposes provided that this notice in its
 * entirety accompanies all copies.
 * 
 * 3. ALL COMMERCIAL USE, AND ALL USE BY FOR PROFIT ENTITIES, IS EXPRESSLY
 * PROHIBITED WITHOUT A LICENSE FROM SIMPLESCALAR, LLC (info@simplescalar.com).
 * 
 * 4. No nonprofit user may place any restrictions on the use of this software,
 * including as modified by the user, by any other authorized user.
 * 
 * 5. Noncommercial and nonprofit users may distribute copies of SimpleScalar
 * in compiled or executable form as set forth in Section 2, provided that
 * either: (A) it is accompanied by the corresponding machine-readable source
 * code, or (B) it is accompanied by a written offer, with no time limit, to
 * give anyone a machine-readable copy of the corresponding source code in
 * return for reimbursement of the cost of distribution. This written offer
 * must permit verbatim duplication by anyone, or (C) it is distributed by
 * someone who received only the executable form, and is accompanied by a
 * copy of the written offer of source code.
 * 
 * 6. SimpleScalar was developed by Todd M. Austin, Ph.D. The tool suite is
 * currently maintained by SimpleScalar LLC (info@simplescalar.com). US Mail:
 * 2395 Timbercrest Court, Ann Arbor, MI 48105.
 * 
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 */


#ifndef OPTIONS_H
#define OPTIONS_H

#include <stdio.h>
#include <stdlib.h>
#ifndef _MSC_VER
#include <unistd.h>
#else /* _MSC_VER */
#define chdir	_chdir
#define getcwd	_getcwd
#endif
#include <string.h>
#include <ctype.h>
#include <float.h>
#include <isis/host.h>
#include <isis/misc.h>

/*
 * This options package allows the user to specify the name, description,
 * location, and default values of program option variables.  In addition,
 * two builtin options are supported:
 *
 *   -config <filename>		# load options from <filename>
 *   -dumpconfig <filename>	# save current option into <filename>
 *
 * NOTE: all user-installed option names must begin with a `-', e.g., `-debug'
 */

/* option variable classes */
enum opt_class_t {
  oc_int = 0,		/* integer option */
  oc_uint,		/* unsigned integer option */
  oc_float,		/* float option */
  oc_double,		/* double option */
  oc_enum,		/* enumeration option */
  oc_flag,		/* boolean option */
  oc_string,		/* string option */
  oc_NUM
};

/* user-specified option definition */
struct opt_opt_t {
  struct opt_opt_t *next;	/* next option */
  char *name;			/* option name, e.g., "-foo:bar" */
  char *desc;			/* option description */
  int nvars;			/* > 1 if var for list options */
  int *nelt;			/* number of elements parsed */
  const char *format;			/* option value print format */
  int print;			/* print option during `-dumpconfig'? */
  int accrue;			/* accrue list across uses */
  int punum;
  enum opt_class_t oc;		/* class of this option */
  union opt_variant_t {
    /* oc == oc_int */
    struct opt_for_int_t {
      int *var;			/* pointer to integer option */
    } for_int;
    /* oc == oc_uint */
    struct opt_for_uint_t {
      unsigned int *var;	/* pointer to unsigned integer option */
    } for_uint;
    /* oc == oc_float */
    struct opt_for_float_t {
      float *var;		/* pointer to float option */
    } for_float;
    /* oc == oc_double */
    struct opt_for_double_t {
      double *var;		/* pointer to double option */
    } for_double;
    /* oc == oc_enum, oc_flag */
    struct opt_for_enum_t {
      int *var;			/* ptr to *int* enum option, NOTE: AN INT */
      char **emap;		/* array of enum strings */
      int *eval;		/* optional array of enum values */
      int emap_sz;		/* number of enum's in arrays */
    } for_enum;
    /* oc == oc_string */
    struct opt_for_string_t {
      char **var;		/* pointer to string pointer option */
    } for_string;
  } variant;
};

/* user-specified argument orphan parser, called when an argument is
   encountered that is not claimed by a user-installed option */
typedef int
(*orphan_fn_t)(int i,		/* index of the orphan'ed argument */
	       int argc,	/* number of program arguments */
	       char **argv);	/* program arguments */

/* an option note, these trail the option list when help or option state
   is printed */
struct opt_note_t {
  struct opt_note_t *next;	/* next option note */
  char *note;			/* option note */
};

/* option database definition */
struct opt_odb_t {
  struct opt_opt_t *options;	/* user-installed options in option database */
  orphan_fn_t orphan_fn;	/* user-specified orphan parser */
  char *header;			/* options header */
  struct opt_note_t *notes;	/* option notes */
};

class option_simple
{
 private:
  void add_option(struct opt_odb_t *, struct opt_opt_t *);
  int bind_to_enum(char *, char **, int *, int, int *);
  char * bind_to_str(int, char **, int *, int);
  int process_option(struct opt_odb_t *, int, int, char **);
  void __opt_process_options(struct opt_odb_t *, int, char **, int);
  void process_file(struct opt_odb_t *, char *, int);
  void print_option_header(struct opt_odb_t *, FILE *);
  void print_option_notes(struct opt_odb_t *, FILE *);
  int opt_null_string(struct opt_opt_t *);
  void print_help(struct opt_opt_t *, FILE *);
  void dump_config(struct opt_odb_t *, char *);
  struct opt_opt_t dumpconfig_opt;
  struct opt_opt_t config_opt;
  struct opt_opt_t *builtin_options;

 public:
  option_simple(){}
  struct opt_odb_t * opt_new(orphan_fn_t orphan_fn);
  
  void opt_delete(struct opt_odb_t *);
  void opt_reg_int(struct opt_odb_t *, char *, char *,
				   int *, int, int, char *, int);
  void opt_reg_int_list(struct opt_odb_t *, char *, char *, int *,
						int, int *, int *, int, char *, int, int);
  void opt_reg_uint(struct opt_odb_t *, char *, char *, unsigned int *,
					unsigned int, int, char *, int);
  void opt_reg_uint_list(struct opt_odb_t *, char *, char *, unsigned int *,
						 int, int *, unsigned int *, int, char *, int, int);
  void opt_reg_float(struct opt_odb_t *, char *, char *, float *,
					 float, int, char *, int);
  void opt_reg_float_list(struct opt_odb_t *, char *, char *, float *,
						  int, int *, float *, int, char *, int, int);
  void opt_reg_double(struct opt_odb_t *, char *, char *, double *,
					  double, int, char *, int);
  void opt_reg_double_list(struct opt_odb_t *, char *, char *, double *,
						   int, int *, double *, int, char *, int, int);
  void opt_reg_enum(struct opt_odb_t *, char *, char *, int *, char *,
					char **, int *, int, int, char *, int);
  void opt_reg_enum_list(struct opt_odb_t *, char *, char *, int *, int,
						 int *, char *, char **, int *, int,
						 int, char *, int, int);
  void opt_reg_flag(struct opt_odb_t *, char *, char *, int *, int, 
					int, char *, int);
  void opt_reg_flag_list(struct opt_odb_t *, char *, char *, int *,
						 int, int *, int *, int, char *, int, int);
  void opt_reg_string(struct opt_odb_t *, char *, char *, char **, char *,
					  int, char *, int);
  void opt_reg_string_list(struct opt_odb_t *, char *, char *, char **,
						   int, int *, char **, int, char *, int, int);
  void opt_process_options(struct opt_odb_t *, int, char **);
  void opt_print_option(struct opt_opt_t *, FILE *);
  void opt_print_options(struct opt_odb_t *, FILE *, int, int);
  void opt_print_help(struct opt_odb_t *, FILE *);
  struct opt_opt_t * opt_find_option(struct opt_odb_t *, char *);
  void opt_reg_header(struct opt_odb_t *, char *);
  void opt_reg_note(struct opt_odb_t *, char *);
};
#endif /* OPTIONS_H */
