/* libexo.c - EXO library main line routines */

/* SimpleScalar(TM) Tool Suite
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 * All Rights Reserved. 
 * 
 * THIS IS A LEGAL DOCUMENT, BY USING SIMPLESCALAR,
 * YOU ARE AGREEING TO THESE TERMS AND CONDITIONS.
 * 
 * No portion of this work may be used by any commercial entity, or for any
 * commercial purpose, without the prior, written permission of SimpleScalar,
 * LLC (info@simplescalar.com). Nonprofit and noncommercial use is permitted
 * as described below.
 * 
 * 1. SimpleScalar is provided AS IS, with no warranty of any kind, express
 * or implied. The user of the program accepts full responsibility for the
 * application of the program and the use of any results.
 * 
 * 2. Nonprofit and noncommercial use is encouraged. SimpleScalar may be
 * downloaded, compiled, executed, copied, and modified solely for nonprofit,
 * educational, noncommercial research, and noncommercial scholarship
 * purposes provided that this notice in its entirety accompanies all copies.
 * Copies of the modified software can be delivered to persons who use it
 * solely for nonprofit, educational, noncommercial research, and
 * noncommercial scholarship purposes provided that this notice in its
 * entirety accompanies all copies.
 * 
 * 3. ALL COMMERCIAL USE, AND ALL USE BY FOR PROFIT ENTITIES, IS EXPRESSLY
 * PROHIBITED WITHOUT A LICENSE FROM SIMPLESCALAR, LLC (info@simplescalar.com).
 * 
 * 4. No nonprofit user may place any restrictions on the use of this software,
 * including as modified by the user, by any other authorized user.
 * 
 * 5. Noncommercial and nonprofit users may distribute copies of SimpleScalar
 * in compiled or executable form as set forth in Section 2, provided that
 * either: (A) it is accompanied by the corresponding machine-readable source
 * code, or (B) it is accompanied by a written offer, with no time limit, to
 * give anyone a machine-readable copy of the corresponding source code in
 * return for reimbursement of the cost of distribution. This written offer
 * must permit verbatim duplication by anyone, or (C) it is distributed by
 * someone who received only the executable form, and is accompanied by a
 * copy of the written offer of source code.
 * 
 * 6. SimpleScalar was developed by Todd M. Austin, Ph.D. The tool suite is
 * currently maintained by SimpleScalar LLC (info@simplescalar.com). US Mail:
 * 2395 Timbercrest Court, Ann Arbor, MI 48105.
 * 
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <limits.h>
#include <errno.h>
#include <assert.h>

#include "host.h"
#include "misc.h"
#include "machine.h"
#include "libexo.h"


void
*yy_flex_realloc(void *ptr, unsigned int size)
{
  return (void *)realloc( (char *) ptr, size );
}

void
*yy_flex_alloc(unsigned int size )
{
  return (void *) malloc( size );
}

libexo::libexo(){
  /* EXO term classes print strings */
  exo_class_str[0] = "integer";
  exo_class_str[1] = "address";
  exo_class_str[2] = "float";
  exo_class_str[3] = "char";
  exo_class_str[4] = "string";
  exo_class_str[5] = "list";
  exo_class_str[6] = "array";
  exo_class_str[7] = "token";
  exo_class_str[8] = "blob";
  
  yy_c_buf_p = (char *) 0;
  yy_current_buffer = 0;
  line = 1;
  yy_init = 1;
  yy_start = 0;

  yy_accept[0] = 0;  yy_accept[1] = 0;  yy_accept[2] = 0;  yy_accept[3] = 0;
  yy_accept[4] = 0;  yy_accept[5] = 26; yy_accept[6] = 24; yy_accept[7] = 22;
  yy_accept[8] = 23; yy_accept[9] = 12; yy_accept[10] = 24; yy_accept[11] = 24;
  yy_accept[12] = 16; yy_accept[13] = 17; yy_accept[14] = 15;
  yy_accept[15] = 12; yy_accept[16] = 24; yy_accept[17] = 12; 
  yy_accept[18] = 5; yy_accept[19] = 5; yy_accept[20] = 20; yy_accept[21] = 11;
  yy_accept[22] = 18; yy_accept[23] = 19; yy_accept[24] = 13; 
  yy_accept[25] = 14; yy_accept[26] = 24; yy_accept[27] = 21; 
  yy_accept[28] = 12; yy_accept[29] = 0; yy_accept[30] = 10; yy_accept[31] = 0;
  yy_accept[32] = 0; yy_accept[33] = 0; yy_accept[34] = 0; yy_accept[35] = 5;
  yy_accept[36] = 5; yy_accept[37] = 7; yy_accept[38] = 2; yy_accept[39] = 8;
  yy_accept[40] = 4; yy_accept[41] = 0; yy_accept[42] = 0; yy_accept[43] = 11;
  yy_accept[44] = 1; yy_accept[45] = 9; yy_accept[46] = 0; yy_accept[47] = 7;
  yy_accept[48] = 0; yy_accept[49] = 0; yy_accept[50] = 6; yy_accept[51] = 3;
  yy_accept[52] = 0; yy_accept[53] = 7; yy_accept[54] = 0; yy_accept[55] = 0;
  yy_accept[56] = 8; yy_accept[57] = 0; yy_accept[58] = 7; yy_accept[59] = 0;

  yy_ec[0] = 0; yy_ec[1] = 1; yy_ec[2] = 1; yy_ec[3] = 1; yy_ec[4] = 1;
  yy_ec[5] = 1; yy_ec[6] = 1; yy_ec[7] = 1; yy_ec[8] = 1; yy_ec[9] = 2;
  yy_ec[10] = 3; yy_ec[11] = 2; yy_ec[12] = 2; yy_ec[13] = 1; yy_ec[14] = 1;
  yy_ec[15] = 1; yy_ec[16] = 1; yy_ec[17] = 1; yy_ec[18] = 1; yy_ec[19] = 1;
  yy_ec[20] = 1; yy_ec[21] = 1;  yy_ec[22] = 1; yy_ec[23] = 1; yy_ec[24] = 1;
  yy_ec[25] = 1; yy_ec[26] = 1; yy_ec[27] = 1; yy_ec[28] = 1; yy_ec[29] = 1;
  yy_ec[30] = 1; yy_ec[31] = 1; yy_ec[32] = 2; yy_ec[33] = 4; yy_ec[34] = 5;
  yy_ec[35] = 6; yy_ec[36] = 7; yy_ec[37] = 8; yy_ec[38] = 9; yy_ec[39] = 10;
  yy_ec[40] = 11; yy_ec[41] = 12; yy_ec[42] = 13; yy_ec[43] = 14;
  yy_ec[44] = 15; yy_ec[45] = 16; yy_ec[46] = 17; yy_ec[47] = 18;
  yy_ec[48] = 19; yy_ec[49] = 20; yy_ec[50] = 20; yy_ec[51] = 20;
  yy_ec[52] = 20; yy_ec[53] = 20; yy_ec[54] = 20; yy_ec[55] = 20;
  yy_ec[56] = 20; yy_ec[57] = 20; yy_ec[58] = 1; yy_ec[59] = 1; yy_ec[60] = 21;
  yy_ec[61] = 1; yy_ec[62] = 22; yy_ec[63] = 23; yy_ec[64] = 24;
  yy_ec[65] = 25; yy_ec[66] = 25; yy_ec[67] = 25; yy_ec[68] = 25;
  yy_ec[69] = 26; yy_ec[70] = 25; yy_ec[71] = 27; yy_ec[72] = 27;
  yy_ec[73] = 27; yy_ec[74] = 27; yy_ec[75] = 27; yy_ec[76] = 27;
  yy_ec[77] = 27; yy_ec[78] = 27; yy_ec[79] = 27; yy_ec[80] =27;
  yy_ec[81] = 27; yy_ec[82] = 27; yy_ec[83] = 27; yy_ec[84] = 27;
  yy_ec[85] = 27; yy_ec[86] = 27; yy_ec[87] = 27; yy_ec[88] = 28;
  yy_ec[89] = 27; yy_ec[90] = 27; yy_ec[91] = 29; yy_ec[92] = 30;
  yy_ec[93] = 31; yy_ec[94] = 32; yy_ec[95] = 27; yy_ec[96] = 1;
  yy_ec[97] = 25; yy_ec[98] = 25; yy_ec[99] = 25; yy_ec[100] = 25;
  yy_ec[101] = 26; yy_ec[102] = 25; yy_ec[103] = 27; yy_ec[104] = 27;
  yy_ec[105] = 27; yy_ec[106] = 27; yy_ec[107] = 27; yy_ec[108] = 27;
  yy_ec[109] = 27; yy_ec[110] = 27; yy_ec[111] = 27; yy_ec[112] = 27;
  yy_ec[113] = 27; yy_ec[114] = 27; yy_ec[115] = 27; yy_ec[116] = 27;
  yy_ec[117] = 27; yy_ec[118] = 27; yy_ec[119] = 27; yy_ec[120] = 28;
  yy_ec[121] = 27; yy_ec[122] = 27; yy_ec[123] = 33; yy_ec[124] = 34;
  yy_ec[125] = 35; yy_ec[126] = 36; yy_ec[127] = 1; yy_ec[128] = 1;
  yy_ec[129] = 1; yy_ec[130] = 1;
  for (int i = 131; i < 256; i++ )
	yy_ec[i] = 1;

  for (int i = 0; i < 37; i++ )
	yy_meta[i] = 1;
  yy_meta[0] = 0; yy_meta[10] = 2; yy_meta[19] = 3; yy_meta[20] = 3;
  yy_meta[25] = 3; yy_meta[26] = 3; yy_meta[27] = 3; yy_meta[28] = 3;

  yy_base[0] = 0; yy_base[1] = 0; yy_base[2] = 0; yy_base[3] = 35;
  yy_base[4] = 60; yy_base[5] = 83; yy_base[6] = 178; yy_base[7] = 178;
  yy_base[8] = 178; yy_base[9] = 83; yy_base[10] = 34; yy_base[11] = 51;
  yy_base[12] = 178; yy_base[13] = 178; yy_base[14] = 178; yy_base[15] = 23;
  yy_base[16] = 25; yy_base[17] = 46; yy_base[18] = 30; yy_base[19] = 48;
  yy_base[20] = 178; yy_base[21] = 0; yy_base[22] = 178; yy_base[23] = 178;
  yy_base[24] = 178; yy_base[25] = 178; yy_base[26] = 50; yy_base[27] = 178;
  yy_base[28] = 0; yy_base[29] = 36; yy_base[30] = 178; yy_base[31] = 49;
  yy_base[32] = 41; yy_base[33] = 43; yy_base[34] = 53; yy_base[35] = 83;
  yy_base[36] = 101; yy_base[37] = 58; yy_base[38] = 0; yy_base[39] = 103;
  yy_base[40] = 111; yy_base[41] = 94; yy_base[42] = 113; yy_base[43] = 0;
  yy_base[44] = 178; yy_base[45] = 178; yy_base[46] = 126; yy_base[47] = 115;
  yy_base[48] = 133; yy_base[49] = 74; yy_base[50] = 85; yy_base[51] = 131;
  yy_base[52] = 105; yy_base[53] = 124; yy_base[54] = 144; yy_base[55] = 135;
  yy_base[56] = 142; yy_base[57] = 146; yy_base[58] = 148; yy_base[59] = 178;
  yy_base[60] = 168; yy_base[61] = 171; yy_base[62] = 174; yy_base[63] = 45;

  yy_def[0] = 0; yy_def[1] = 59; yy_def[2] = 1; yy_def[3] = 60;
  yy_def[4] = 60; yy_def[5] = 59; yy_def[6] = 59; yy_def[7] = 59;
  yy_def[8] = 59; yy_def[9] = 59; yy_def[10] = 61; yy_def[11] = 62;
  yy_def[12] = 59; yy_def[13] = 59; yy_def[14] = 59; yy_def[15] = 9;
  yy_def[16] = 59; yy_def[17] = 9; yy_def[18] = 59; yy_def[19] = 59;
  yy_def[20] = 59; yy_def[21] = 63; yy_def[22] = 59; yy_def[23] = 59;
  yy_def[24] = 59; yy_def[25] = 59; yy_def[26] = 59; yy_def[27] = 59;
  yy_def[28] = 9; yy_def[29] = 61; yy_def[30] = 59; yy_def[31] = 61;
  yy_def[32] = 62; yy_def[33] = 62; yy_def[34] = 59; yy_def[35] = 59;
  yy_def[36] = 59; yy_def[37] = 59; yy_def[38] = 9; yy_def[39] = 59;
  yy_def[40] = 59; yy_def[41] = 59; yy_def[42] = 59; yy_def[43] = 63;
  yy_def[44] = 59; yy_def[45] = 59; yy_def[46] = 59; yy_def[47] = 59;
  yy_def[48] = 59; yy_def[49] = 59; yy_def[50] = 59; yy_def[51] = 59;
  yy_def[52] = 59; yy_def[53] = 59; yy_def[54] = 59; yy_def[55] = 59;
  yy_def[56] = 59; yy_def[57] = 59; yy_def[58] = 59; yy_def[59] = 0;
  yy_def[60] = 59; yy_def[61] = 59; yy_def[62] = 59; yy_def[63] = 59;

  yy_nxt[0] = 0; yy_nxt[1] = 6; yy_nxt[2] = 7; yy_nxt[3] = 8;
  yy_nxt[4] = 9; yy_nxt[5] = 10; yy_nxt[6] = 9; yy_nxt[7] = 9;
  yy_nxt[8] = 9; yy_nxt[9] = 9; yy_nxt[10] = 11; yy_nxt[11] = 12;
  yy_nxt[12] = 13; yy_nxt[13] = 9; yy_nxt[14] = 9; yy_nxt[15] = 14;
  yy_nxt[16] = 15; yy_nxt[17] = 16; yy_nxt[18] = 17; yy_nxt[19] = 18;
  yy_nxt[20] = 19; yy_nxt[21] = 20; yy_nxt[22] = 6; yy_nxt[23] = 9;
  yy_nxt[24] = 9; yy_nxt[25] = 21; yy_nxt[26] = 21; yy_nxt[27] = 21;
  yy_nxt[28] = 21; yy_nxt[29] = 22; yy_nxt[30] = 6; yy_nxt[31] = 23;
  yy_nxt[32] = 9; yy_nxt[33] = 24; yy_nxt[34] = 9; yy_nxt[35] = 25;
  yy_nxt[36] = 9; yy_nxt[37] = 7; yy_nxt[38] = 8; yy_nxt[39] = 30;
  yy_nxt[40] = 34; yy_nxt[41] = 30; yy_nxt[42] = 35; yy_nxt[43] = 36;
  yy_nxt[44] = 37; yy_nxt[45] = 37; yy_nxt[46] = 59; yy_nxt[47] = 39;
  yy_nxt[48] = 43; yy_nxt[49] = 40; yy_nxt[50] = 40; yy_nxt[51] = 45;
  yy_nxt[52] = 59; yy_nxt[53] = 32; yy_nxt[54] = 26; yy_nxt[55] = 26;
  yy_nxt[56] = 41; yy_nxt[57] = 27; yy_nxt[58] = 42; yy_nxt[59] = 38;
  yy_nxt[60] = 26; yy_nxt[61] = 26; yy_nxt[62] = 7; yy_nxt[63] = 8;
  yy_nxt[64] = 31; yy_nxt[65] = 39; yy_nxt[66] = 31; yy_nxt[67] = 36;
  yy_nxt[68] = 36; yy_nxt[69] = 44; yy_nxt[70] = 44; yy_nxt[71] = 33;
  yy_nxt[72] = 37; yy_nxt[73] = 37; yy_nxt[74] = 41; yy_nxt[75] = 44;
  yy_nxt[76] = 44; yy_nxt[77] = 37; yy_nxt[78] = 37; yy_nxt[79] = 26;
  yy_nxt[80] = 26; yy_nxt[81] = 33; yy_nxt[82] = 27; yy_nxt[83] = 59;
  yy_nxt[84] = 46; yy_nxt[85] = 26; yy_nxt[86] = 26; yy_nxt[87] = 28;
  yy_nxt[88] = 59; yy_nxt[89] = 28; yy_nxt[90] = 28; yy_nxt[91] = 28;
  yy_nxt[92] = 28; yy_nxt[93] = 50; yy_nxt[94] = 50; yy_nxt[95] = 59;
  yy_nxt[96] = 28; yy_nxt[97] = 28; yy_nxt[98] = 59; yy_nxt[99] = 28;
  yy_nxt[100] = 39; yy_nxt[101] = 28; yy_nxt[102] = 40; yy_nxt[103] = 40;
  yy_nxt[104] = 50; yy_nxt[105] = 50; yy_nxt[106] = 28; yy_nxt[107] = 28;
  yy_nxt[108] = 49; yy_nxt[109] = 41; yy_nxt[110] = 49; yy_nxt[111] = 42;
  yy_nxt[112] = 59; yy_nxt[113] = 50; yy_nxt[114] = 50; yy_nxt[115] = 28;
  yy_nxt[116] = 59; yy_nxt[117] = 28; yy_nxt[118] = 39; yy_nxt[119] = 28;
  yy_nxt[120] = 36; yy_nxt[121] = 36; yy_nxt[122] = 47; yy_nxt[123] = 47;
  yy_nxt[124] = 53; yy_nxt[125] = 53; yy_nxt[126] = 59; yy_nxt[127] = 41;
  yy_nxt[128] = 39; yy_nxt[129] = 48; yy_nxt[130] = 40; yy_nxt[131] = 40;
  yy_nxt[132] = 51; yy_nxt[133] = 51; yy_nxt[134] = 47; yy_nxt[135] = 47;
  yy_nxt[136] = 59; yy_nxt[137] = 41; yy_nxt[138] = 51; yy_nxt[139] = 51;
  yy_nxt[140] = 52; yy_nxt[141] = 54; yy_nxt[142] = 52; yy_nxt[143] = 53;
  yy_nxt[144] = 53; yy_nxt[145] = 53; yy_nxt[146] = 53; yy_nxt[147] = 55;
  yy_nxt[148] = 59; yy_nxt[149] = 55; yy_nxt[150] = 51; yy_nxt[151] = 51;
  yy_nxt[152] = 56; yy_nxt[153] = 56; yy_nxt[154] = 56; yy_nxt[155] = 56;
  yy_nxt[156] = 51; yy_nxt[157] = 51; yy_nxt[158] = 57; yy_nxt[159] = 59;
  yy_nxt[160] = 57; yy_nxt[161] = 56; yy_nxt[162] = 56; yy_nxt[163] = 58;
  yy_nxt[164] = 58; yy_nxt[165] = 58; yy_nxt[166] = 58; yy_nxt[167] = 58;
  yy_nxt[168] = 58; yy_nxt[169] = 6; yy_nxt[170] = 6; yy_nxt[171] = 6;
  yy_nxt[172] = 29; yy_nxt[173] = 29; yy_nxt[174] = 29; yy_nxt[175] = 32;
  yy_nxt[176] = 59; yy_nxt[177] = 32; yy_nxt[178] = 5;
  for (int i = 179; i < 215; i++ )
	yy_nxt[i] = 59;

  yy_chk[0] = 0;
  for (int i = 1; i < 37 ; i++ )
	yy_chk[i] = 1;
  yy_chk[37] = 3; yy_chk[38] = 3; yy_chk[39] = 10; yy_chk[40] = 15;
  yy_chk[41] = 29; yy_chk[42] = 15; yy_chk[43] = 15; yy_chk[44] = 16;
  yy_chk[45] = 16; yy_chk[46] = 33; yy_chk[47] = 18; yy_chk[48] = 63;
  yy_chk[49] = 18; yy_chk[50] = 18; yy_chk[51] = 32; yy_chk[52] = 31;
  yy_chk[53] = 33; yy_chk[54] = 3; yy_chk[55] = 3; yy_chk[56] = 18;
  yy_chk[57] = 3; yy_chk[58] = 18; yy_chk[59] = 17; yy_chk[60] = 3;
  yy_chk[61] = 3; yy_chk[62] = 4; yy_chk[63] = 4; yy_chk[64] = 10;
  yy_chk[65] = 19; yy_chk[66] = 29; yy_chk[67] = 19; yy_chk[68] = 19;
  yy_chk[69] = 26; yy_chk[70] = 26; yy_chk[71] = 32; yy_chk[72] = 34;
  yy_chk[73] = 34; yy_chk[74] = 19; yy_chk[75] = 26; yy_chk[76] = 26;
  yy_chk[77] = 37; yy_chk[78] = 37; yy_chk[79] = 4; yy_chk[80] = 4;
  yy_chk[81] = 11; yy_chk[82] = 4; yy_chk[83] = 5; yy_chk[84] = 37;
  yy_chk[85] = 4; yy_chk[86] = 4; yy_chk[87] = 9; yy_chk[88] = 0;
  yy_chk[89] = 9; yy_chk[90] = 9; yy_chk[91] = 9; yy_chk[92] = 9;
  yy_chk[93] = 49; yy_chk[94] = 49; yy_chk[95] = 0; yy_chk[96] = 9;
  yy_chk[97] = 9; yy_chk[98] = 0; yy_chk[99] = 9; yy_chk[100] = 35;
  yy_chk[101] = 9; yy_chk[102] = 35; yy_chk[103] = 35; yy_chk[104] = 50;
  yy_chk[105] = 50; yy_chk[106] = 9; yy_chk[107] = 9; yy_chk[108] = 41;
  yy_chk[109] = 35; yy_chk[110] = 41; yy_chk[111] = 35; yy_chk[112] = 0;
  yy_chk[113] = 41; yy_chk[114] = 41; yy_chk[115] = 9; yy_chk[116] = 0;
  yy_chk[117] = 9; yy_chk[118] = 36; yy_chk[119] = 9; yy_chk[120] = 36;
  yy_chk[121] = 36; yy_chk[122] = 39; yy_chk[123] = 39; yy_chk[124] = 52;
  yy_chk[125] = 52; yy_chk[126] = 0; yy_chk[127] = 36; yy_chk[128] = 40;
  yy_chk[129] = 39; yy_chk[130] = 40; yy_chk[131] = 40; yy_chk[132] = 42;
  yy_chk[133] = 42; yy_chk[134] = 47; yy_chk[135] = 47; yy_chk[136] = 0;
  yy_chk[137] = 40; yy_chk[138] = 42; yy_chk[139] = 42; yy_chk[140] = 46;
  yy_chk[141] = 47; yy_chk[142] = 46; yy_chk[143] = 53; yy_chk[144] = 53;
  yy_chk[145] = 46; yy_chk[146] = 46; yy_chk[147] = 48; yy_chk[148] = 0;
  yy_chk[149] = 48; yy_chk[150] = 51; yy_chk[151] = 51; yy_chk[152] = 48;
  yy_chk[153] = 48; yy_chk[154] = 55; yy_chk[155] = 55; yy_chk[156] = 51;
  yy_chk[157] = 51; yy_chk[158] = 54; yy_chk[159] = 0; yy_chk[160] = 54;
  yy_chk[161] = 56; yy_chk[162] = 56; yy_chk[163] = 54; yy_chk[164] = 54;
  yy_chk[165] = 57; yy_chk[166] = 57; yy_chk[167] = 58; yy_chk[168] = 58;
  yy_chk[169] = 60; yy_chk[170] = 60; yy_chk[171] = 60; yy_chk[172] = 61;
  yy_chk[173] = 61; yy_chk[174] = 61; yy_chk[175] = 62; yy_chk[176] = 0;
  yy_chk[177] = 62;
  for (int i = 178; i < 215 ; i++ )
	yy_chk[i] = 59; 
}

/* return the value of an escape sequence, ESCAPE is a pointer to the first
   character following '\', sets NEXT to first character after escape */
/* A2.5.2 */
int
libexo::intern_escape(char *esc, char **next)
{
  int c, value, empty, count;

  switch (c = *esc++) {
  case 'x':
    /* \xhh hex value */
    value = 0;
    empty = TRUE;
    while (1)
      {
        c = *esc++;
        if (!(c >= 'a' && c <= 'f')
            && !(c >= 'A' && c <= 'F')
            && !(c >= '0' && c <= '9'))
          {
            esc--;
            break;
          }
        value *=16;
        if (c >= 'a' && c <= 'f')
          value += c - 'a' + 10;
        if (c >= 'A' && c <= 'F')
          value += c - 'A' + 10;
        if (c >= '0' && c <= '9')
          value += c - '0';
        empty = FALSE;
      }
    if (empty)
      fatal("\\x used with no trailing hex digits");
    break;

  case '0': case '1': case '2': case '3':
  case '4': case '5': case '6': case '7':
    /* \ooo octal value */
    value = 0;
    count = 0;
    while ((c <= '7') && (c >= '0') && (count++ < 3))
      {
        value = (value * 8) + (c - '0');
        c = *esc++;
      }
    esc--;
    break;

  case '\\':
  case '\'':
  case '"':
    value = c;
    break;

  case 'n':
    value = '\n';
    break;

  case 't':
    value = '\t';
    break;

  case 'r':
    value = '\r';
    break;

  case 'f':
    value = '\f';
    break;

  case 'b':
    value = '\b';
    break;

  case 'a':
    value = '\a';
    break;

  case 'v':
    value = '\v';
    break;

  case '?':
    value = c;
    break;

  case '(':
  case '{':
  case '[':
  case '%':
    value = c;
    warn("non-ANSI escape sequence `\\%c'", c);
    break;

  default:
    fatal("unknown escape, '\\' followed by char %x (`%c')", (int)c, c);
  }

  if (*next)
    *next = esc;

  return value;
}

/* return the value of an character literal sequence */
/* A2.5.2 */
int
libexo::intern_char(char *s, char **next)
{
  unsigned char value;

  if (s[0] != '\'' || s[strlen(s)-1] != '\'')
    panic("mal-formed string constant");

  if (s[1] != '\\')
    {
      value = (unsigned)s[1];
      if (s[2] != '\'')
		panic("mal-formed string constant");
      if (next)
		*next = s + 2;
    }
  else
    {
      /* escaped char constant */
      value = intern_escape(s+2, next);
    }

  /* map to a signed char value */
  value = (signed int)((unsigned char)((unsigned char)((unsigned int)value)));

  if (UCHAR_MAX < value)
    fatal("character constant out of range");

  return value;
}

void
libexo::print_char(unsigned char c, FILE *stream)
{
  switch (c)
    {
    case '\n':
      fprintf(stream, "\\n");
      break;

    case '\\':
      fprintf(stream, "\\\\");
      break;

    case '\'':
      fprintf(stream, "\\'");
      break;

    case '\t':
      fprintf(stream, "\\t");
      break;

    case '\r':
      fprintf(stream, "\\r");
      break;

    case '\f':
      fprintf(stream, "\\f");
      break;

    case '\b':
      fprintf(stream, "\\b");
      break;

    case '\a':
      fprintf(stream, "\\a");
      break;

    case '\v':
      fprintf(stream, "\\v");
      break;

    default:
      if (isprint(c))
		fprintf(stream, "%c", c);
      else
		fprintf(stream, "\\x%02x", c);
    }
}

/* expand all escapes in string STR, return pointer to allocation w/ result */
char *
libexo::intern_string(char *str)
{
  char *s, *istr;

  /* resulting string cannot be longer than STR */
  s = istr = (char *)malloc(strlen(str)+1);

  if (!str || !*str || *str != '\"') /* " */
    panic("mal-formed string constant");

  /* skip `"' */ /* " */
  str++;

  while (*str)
    {
      if (*str == '\\')
        *s++ = intern_escape(str+1, &str);
      else
        {
          /* A2.6 */
          if (*str == '\n')
            warn("ANSI C forbids newline in character constant");
          /* A2.6 */
          if (*str == '"' && str[1] != '\0')
            panic("encountered `\"' embedded in string constant");

          if (*str != '\"') /* " */
            *s++ = *str;
          str++;
        }
    }
  *s = '\0';
  return istr;
}

void
libexo::print_string(unsigned char *s, FILE *stream)
{
  while (*s)
    {
      print_char(*s, stream);
      s++;
    }
}

/* bogus token value */
#define TOKEN_BOGON		0

static int token_id = TOKEN_BOGON + 1;

#define TOKEN_HASH_SIZE		1024
struct exo_token_t *token_hash[TOKEN_HASH_SIZE];

/* hash a string */
unsigned long
libexo::hash_str(char *s)
{
  unsigned h = 0;
  while (*s)
    h = (h << 1) + *s++;
  return (h % TOKEN_HASH_SIZE);
}

/* intern token TOKEN_STR */
struct exo_token_t *
libexo::exo_intern(char *token_str)		/* string to intern */
{
  int index;
  struct exo_token_t *ent;

  index = hash_str(token_str);

  for (ent=token_hash[index]; ent != NULL; ent=ent->next) {
	if (!strcmp(token_str, ent->str)) {
	  /* got a match, return token entry */
	  return ent;
	}
  }
  
  /* not found, create a new entry */
  ent = (struct exo_token_t *)calloc(1, sizeof(struct exo_token_t));
  if (!ent)
    fatal("out of virtual memory");

  ent->str = mystrdup(token_str);
  ent->token = token_id++;
  ent->next = token_hash[index];
  token_hash[index] = ent;

  return ent;
}

/* intern token TOKEN_STR as value TOKEN */
struct exo_token_t *
libexo::exo_intern_as(char *token_str,		/* string to intern */
					  int token)		/* internment value */
{
  struct exo_token_t *ent;

#if 0
  if (token_id > token)
    fatal("token value is already in use");
#endif

  ent = exo_intern(token_str);
  /* overide the default value */
  ent->token = token;

#if 0
  if (ent->token != token)
    fatal("symbol `%s' was previously interned", token_str);
#endif

  return ent;
}

/* allocate an EXO node, fill in its type */
struct exo_term_t *
libexo::exo_alloc(enum exo_class_t ec)
{
  struct exo_term_t *exo;

  exo = (struct exo_term_t *)calloc(1, sizeof(struct exo_term_t));
  if (!exo)
    fatal("out of virtual memory");
  exo->next = NULL;
  exo->ec = ec;

  return exo;
}


/*
 * create a new EXO term, usage:
 *
 *	exo_new(ec_integer, (exo_integer_t)<int>);
 *	exo_new(ec_address, (exo_integer_t)<int>);
 *	exo_new(ec_float, (exo_float_t)<float>);
 *	exo_new(ec_char, (int)<char>);
 *      exo_new(ec_string, "<string>");
 *      exo_new(ec_list, <list_ent>..., NULL);
 *      exo_new(ec_array, <size>, <array_ent>..., NULL);
 *	exo_new(ec_token, "<token>"); 
 */
struct exo_term_t *
libexo::exo_new(enum exo_class_t ec, ...)
{
  struct exo_term_t *exo;
  va_list v;
  va_start(v, ec);

  exo = exo_alloc(ec);
  switch (ec)
    {
    case ec_integer:
      exo->as_integer.val = va_arg(v, exo_integer_t);
      break;

    case ec_address:
      exo->as_address.val = va_arg(v, exo_address_t);
      break;

    case ec_float:
      exo->as_float.val = va_arg(v, exo_float_t);
      break;

    case ec_char:
      exo->as_char.val = va_arg(v, int);
      break;

    case ec_string:
      {
		char *str;

		str = va_arg(v, char *);
		exo->as_string.str = (unsigned char *)mystrdup(str);
      }
      break;

    case ec_list:
      {
		struct exo_term_t *ent;

		exo->as_list.head = NULL;
		do {
		  ent = va_arg(v, struct exo_term_t *);
		  exo->as_list.head = exo_chain(exo->as_list.head, ent);
		} while (ent != NULL);
      }
      break;

    case ec_array:
      {
		int i;
		struct exo_term_t *ent;

		exo->as_array.size = va_arg(v, int);
		exo->as_array.array = (struct exo_term_t **)
		  calloc(exo->as_array.size, sizeof(struct exo_term_t *));
		if (!exo->as_array.array)
		  fatal("out of virtual memory");
		i = 0;
		do {
		  ent = va_arg(v, struct exo_term_t *);
		  if (ent != NULL)
			{
			  if (i == exo->as_array.size)
				fatal("array constructor overflow");
			  SET_EXO_ARR(exo, i, ent);
			}
		  i++;
		} while (ent != NULL);
      }
      break;

    case ec_token:
      {
		char *str;

		str = va_arg(v, char *);
		exo->as_token.ent = exo_intern(str);
      }
      break;

    case ec_blob:
      {
		unsigned size;
		unsigned char *data;

		size = va_arg(v, unsigned);
		data = va_arg(v, unsigned char *);

		exo->as_blob.size = size;
		exo->as_blob.data = (unsigned char *)malloc(size);
		if (data != NULL)
		  memcpy(exo->as_blob.data, data, size);
		else
		  memset(exo->as_blob.data, 0, size);
      }
      break;

    case ec_null:
      break;

    default:
      panic("bogus EXO class");
    }

  va_end(v);
  return exo;
}

/* release an EXO term */
void
libexo::exo_delete(struct exo_term_t *exo)
{
  exo->next = NULL;

  switch (exo->ec)
    {
    case ec_integer:
      /* no extra storage */
      exo->as_integer.val = 0;
      break;

    case ec_address:
      /* no extra storage */
      exo->as_address.val = 0;
      break;

    case ec_float:
      /* no extra storage */
      exo->as_float.val = 0.0;
      break;

    case ec_char:
      /* no extra storage */
      exo->as_char.val = '\0';
      break;

    case ec_string:
      free(exo->as_string.str);
      exo->as_string.str = NULL;
      break;

    case ec_list:
      {
		struct exo_term_t *ent, *next_ent;

		for (ent=exo->as_list.head; ent != NULL; ent = next_ent)
		  {
			next_ent = ent->next;
			exo_delete(ent);
		  }
		exo->as_list.head = NULL;
      }
      break;

    case ec_array:
      {
		int i;

		for (i=0; i < exo->as_array.size; i++)
		  {
			if (exo->as_array.array[i] != NULL)
			  exo_delete(exo->as_array.array[i]);
		  }
		free(exo->as_array.array);
		exo->as_array.array = NULL;
		exo->as_array.size = 0;
      }
      break;

    case ec_token:
      /* no extra storage */
      exo->as_token.ent = NULL;
      break;

    case ec_blob:
      /* free the blob data */
      free(exo->as_blob.data);
      exo->as_blob.data = NULL;
      break;

    case ec_null:
      /* no extra storage */
      break;

    default:
      panic("bogus EXO class");
    }
  exo->ec = (enum exo_class_t)0;

  /* release the node */
  free(exo);
}

/* chain two EXO lists together, FORE is attached on the end of AFT */
struct exo_term_t *
libexo::exo_chain(struct exo_term_t *fore, struct exo_term_t *aft)
{
  struct exo_term_t *exo, *prev;

  if (!fore && !aft)
    return NULL;

  if (!fore)
    return aft;

  /* find the tail of FORE */
  for (prev=NULL,exo=fore; exo != NULL; prev=exo,exo=exo->next)
    /* nada */;
  assert(prev);

  /* link onto the tail of FORE */
  prev->next = aft;

  return fore;
}

/* copy an EXO node */
struct exo_term_t *
libexo::exo_copy(struct exo_term_t *exo)
{
  struct exo_term_t *new_exo;

  /* NULL copy */
  if (!exo)
    return NULL;

  new_exo = exo_alloc(exo->ec);
  *new_exo = *exo;

  /* the next link is always blown away on a copy */
  new_exo->next = NULL;

  switch (new_exo->ec)
    {
    case ec_integer:
    case ec_address:
    case ec_float:
    case ec_char:
    case ec_string:
    case ec_list:
    case ec_token:
      /* no internal parts to copy */
      break;

    case ec_array:
      {
		int i;

		/* copy the array */
		new_exo->as_array.array = (struct exo_term_t **)
		  calloc(new_exo->as_array.size, sizeof(struct exo_term_t *));

		for (i=0; i<new_exo->as_array.size; i++)
		  {
			SET_EXO_ARR(new_exo, i, EXO_ARR(exo, i));
		  }
      }
      break;

    case ec_blob:
      new_exo->as_blob.data = (unsigned char *)malloc(new_exo->as_array.size);
      memcpy(new_exo->as_blob.data, exo->as_blob.data, new_exo->as_array.size);
      break;

    default:
      panic("bogus EXO class");
    }

  return new_exo;
}

/* deep copy an EXO structure */
struct exo_term_t *
libexo::exo_deepcopy(struct exo_term_t *exo)
{
  struct exo_term_t *new_exo;

  /* NULL copy */
  if (!exo)
    return NULL;

  new_exo = exo_copy(exo);
  switch (new_exo->ec)
    {
    case ec_integer:
    case ec_address:
    case ec_float:
    case ec_char:
    case ec_token:
      /* exo_copy() == exo_deepcopy() for these node classes */
      break;

    case ec_string:
      /* copy the referenced string */
      new_exo->as_string.str =
		(unsigned char *)mystrdup((char *)exo->as_string.str);
      break;

    case ec_list:
      /* copy all list elements */
      {
		struct exo_term_t *elt, *new_elt, *new_list;

		new_list = NULL;
		for (elt=new_exo->as_list.head; elt != NULL; elt=elt->next)
		  {
			new_elt = exo_deepcopy(elt);
			new_list = exo_chain(new_list, new_elt);
		  }
		new_exo->as_list.head = new_list;
      }
      break;

    case ec_array:
      /* copy all array elements */
      {
		int i;

		for (i=0; i<new_exo->as_array.size; i++)
		  {
			SET_EXO_ARR(new_exo, i, exo_deepcopy(EXO_ARR(exo, i)));
		  }
      }
      break;

    case ec_blob:
      new_exo->as_blob.data = (unsigned char *)malloc(new_exo->as_array.size);
      memcpy(new_exo->as_blob.data, exo->as_blob.data, new_exo->as_array.size);
      break;

    default:
      panic("bogus EXO class");
    }

  return new_exo;
}

/* print an EXO term */
void
libexo::exo_print(struct exo_term_t *exo, FILE *stream)
{
  if (!stream)
    stream = stderr;

  switch (exo->ec)
    {
    case ec_integer:
      if (sizeof(exo_integer_t) == 4)
		myfprintf(stream, "%u", exo->as_integer.val);
      else
		myfprintf(stream, "%lu", exo->as_integer.val);
      break;

    case ec_address:
      if (sizeof(exo_address_t) == 4)
		myfprintf(stream, "0x%x", exo->as_integer.val);
      else
		myfprintf(stream, "0x%lx", exo->as_integer.val);
      break;

    case ec_float:
      fprintf(stream, "%f", exo->as_float.val);
      break;

    case ec_char:
      fprintf(stream, "'");
      print_char(exo->as_char.val, stream);
      fprintf(stream, "'");
      break;

    case ec_string:
      fprintf(stream, "\"");
      print_string(exo->as_string.str, stream);
      fprintf(stream, "\"");
      break;

    case ec_list:
      {
		struct exo_term_t *ent;

		fprintf(stream, "(");
		for (ent=exo->as_list.head; ent != NULL; ent=ent->next)
		  {
			exo_print(ent, stream);
			if (ent->next)
			  fprintf(stream, ", ");
		  }
		fprintf(stream, ")");
      }
      break;

    case ec_array:
      {
		int i, last;

		/* search for last first non-NULL entry */
		for (last=exo->as_array.size-1; last >= 0; last--)
		  {
			if (EXO_ARR(exo, last) != NULL)
			  break;
		  }
		/* LAST == index of last non-NULL array entry */

		fprintf(stream, "{%d}[", exo->as_array.size);
		for (i=0; i<exo->as_array.size && i <= last; i++)
		  {
			if (exo->as_array.array[i] != NULL)
			  exo_print(exo->as_array.array[i], stream);
			else
			  fprintf(stream, " ");
			if (i != exo->as_array.size-1 && i != last)
			  fprintf(stream, ", ");
		  }
		fprintf(stream, "]");
      }
      break;

    case ec_token:
      fprintf(stream, "%s", exo->as_token.ent->str);
      break;

    case ec_blob:
      {
		int i, cr;

		fprintf(stream, "{%d}<\n", exo->as_blob.size);
		for (i=0; i < exo->as_blob.size; i++)
		  {
			cr = FALSE;
			if (i != 0 && (i % 38) == 0)
			  {
				fprintf(stream, "\n");
				cr = TRUE;
			  }
			fprintf(stream, "%02x", exo->as_blob.data[i]);
		  }
		if (!cr)
		  fprintf(stream, "\n");
		fprintf(stream, ">");
      }
      break;

    default:
      panic("bogus EXO class");
    }
}


void
libexo::exo_err(char *err)
{
  fprintf(stderr, "EXO parse error: line %d: %s\n", line, err);
  exit(1);
}


/* read one EXO term from STREAM */
struct exo_term_t *
libexo::exo_read(FILE *stream)
{
  int tok;
  char tok_buf[1024], *endp;
  struct exo_term_t *ent = NULL;

  /* make sure we have a valid stream */
  if (!stream)
    stream = stdin;
  yy_setstream(stream);

  /* make local copies of everything, allows arbitrary recursion */
  tok = yylex();
  strcpy(tok_buf, yytext);

  switch (tok)
    {
    case lex_integer:
      {
		exo_integer_t int_val;

		/* attempt integer conversion */
		errno = 0;
#ifdef HOST_HAS_QWORD
		int_val = myatoq(tok_buf, &endp, /* parse base */10);
#else /* !HOST_HAS_QWORD */
		int_val = strtoul(tok_buf, &endp, /* parse base */10);
#endif /* HOST_HAS_QWORD */
		if (!errno && !*endp)
		  {
			/* good conversion */
			ent = exo_new(ec_integer, int_val);
		  }
		else
		  exo_err("cannot parse integer literal");
      }
      break;

    case lex_address:
      {
		exo_address_t addr_val;

		/* attempt address conversion */
		errno = 0;
#ifdef HOST_HAS_QWORD
		addr_val = myatoq(tok_buf, &endp, /* parse base */16);
#else /* !HOST_HAS_QWORD */
		addr_val = strtoul(tok_buf, &endp, /* parse base */16);
#endif /* HOST_HAS_QWORD */
		if (!errno && !*endp)
		  {
			/* good conversion */
			ent = exo_new(ec_address, addr_val);
		  }
		else
		  exo_err("cannot parse address literal");
      }
      break;

    case lex_float:
      {
		exo_float_t float_val;

		/* attempt double conversion */
		errno = 0;
		float_val = strtod(tok_buf, &endp);
		if (!errno && !*endp)
		  {
			/* good conversion */
			ent = exo_new(ec_float, float_val);
		  }
		else
		  exo_err("cannot parse floating point literal");
      }
      break;

    case lex_char:
      {
		int c;

		c = intern_char(tok_buf, &endp);
		if (!endp)
		  exo_err("cannot convert character literal");
		ent = exo_new(ec_char, c);
      }
      break;

    case lex_string:
      {
		char *s;

		s = intern_string(tok_buf);
		ent = exo_new(ec_string, s);
		free(s);
      }
      break;

    case lex_token:
      ent = exo_new(ec_token, tok_buf);
      break;

    case lex_byte:
      exo_err("unexpected blob byte encountered");
      break;

    case '(':
      {
		struct exo_term_t *elt;

		ent = exo_new(ec_list, NULL);

		if (yy_nextchar() != ')')
		  {
			/* not an empty list */
			do {
			  elt = exo_read(stream);
			  if (!elt)
				exo_err("unexpected end-of-file");
			  ent->as_list.head =
				exo_chain(ent->as_list.head, elt);

			  /* consume optional commas */
			  if (yy_nextchar() == ',')
				yylex();
			} while (yy_nextchar() != ')');
		  }

		/* read tail delimiter */
		tok = yylex();
		if (tok != ')')
		  exo_err("expected ')'");
      }
      break;

    case ')':
      exo_err("unexpected ')' encountered");
      break;

    case '<':
      exo_err("unexpected '<' encountered");
      break;

    case '>':
      exo_err("unexpected '>' encountered");
      break;

    case '{':
      {
		int cnt, size;
		struct exo_term_t *elt;

		/* get the size */
		elt = exo_read(stream);
		if (!elt || elt->ec != ec_integer)
		  exo_err("badly formed array size");

		/* record the size of the array/blob */
		size = (int)elt->as_integer.val;

		/* done with the EXO integer */
		exo_delete(elt);

		/* read the array delimiters */
		tok = yylex();
		if (tok != '}')
		  exo_err("expected '}'");

		tok = yylex();
		switch (tok)
		  {
		  case '[': /* array definition */
			/* allocate an array definition */
			ent = exo_new(ec_array, size, NULL);

			/* read until array is full or tail delimiter encountered */
			if (yy_nextchar() != ']')
			  {
				/* not an empty array */
				cnt = 0;
				do {
				  if (cnt == ent->as_array.size)
					exo_err("too many initializers for array");

				  /* NULL element? */
				  if (yy_nextchar() == ',')
					{
					  elt = NULL;
					}
				  else
					{
					  elt = exo_read(stream);
					  if (!elt)
						exo_err("unexpected end-of-file");
					}
				  SET_EXO_ARR(ent, cnt, elt);
				  cnt++;

				  /* consume optional commas */
				  if (yy_nextchar() == ',')
					yylex();
				} while (yy_nextchar() != ']');
			  }

			/* read tail delimiter */
			tok = yylex();
			if (tok != ']')
			  exo_err("expected ']'");
			break;

		  case '<': /* blob definition */
			/* allocate an array definition */
			ent = exo_new(ec_blob, size, /* zero contents */NULL);

			/* read until blob is full */
			if (yy_nextchar() != '>')
			  {
				unsigned int byte_val;

				/* not an empty array */
				cnt = 0;
				for (;;) {
				  /* read next blob byte */
				  tok = yylex();

				  if (tok == lex_byte)
					{
					  if (cnt == ent->as_blob.size)
						exo_err("too many initializers for blob");

					  /* attempt hex conversion */
					  errno = 0;
					  byte_val = strtoul(yytext, &endp, /* parse base */16);
					  if (errno != 0 || *endp != '\0')
						exo_err("cannot parse blob byte literal");
					  if (byte_val > 255)
						panic("bogus byte value");
					  ent->as_blob.data[cnt] = byte_val;
					  cnt++;
					}
				  else if (tok == '>')
					break;
				  else
					exo_err("unexpected character in blob");
				}
			  }

#if 0 /* zero tail is OK... */
			if (cnt != ent->as_blob.size)
			  exo_err("not enough initializers for blob");
#endif
			break;

		  default:
			exo_err("expected '[' or '<'");
		  }
      }
      break;

    case '}':
      exo_err("unexpected '}' encountered");
      break;

    case ',':
      exo_err("unexpected ',' encountered");
      break;

    case '[':
      {
		int i, cnt;
		struct exo_term_t *list, *elt, *next_elt;

		/* compute the array size */
		list = NULL;
		if (yy_nextchar() == ']')
		  exo_err("unsized array has no initializers");

		cnt = 0;
		do {
		  /* NULL element? */
		  if (yy_nextchar() == ',')
			{
			  elt = exo_new(ec_null);
			}
		  else
			{
			  elt = exo_read(stream);
			  if (!elt)
				exo_err("unexpected end-of-file");
			}
		  cnt++;
		  list = exo_chain(list, elt);

		  /* consume optional commas */
		  if (yy_nextchar() == ',')
			yylex();
		} while (yy_nextchar() != ']');

		/* read tail delimiter */
		tok = yylex();
		if (tok != ']')
		  exo_err("expected ']'");

		/* create the array */
		assert(cnt > 0);
		ent = exo_new(ec_array, cnt, NULL);

		/* fill up the array */
		for (i=0,elt=list; i<cnt; i++,elt=next_elt)
		  {
			assert(elt != NULL);
			next_elt = elt->next;
			if (elt->ec == ec_null)
			  {
				SET_EXO_ARR(ent, cnt, NULL);
				exo_delete(ent);
			  }
			else
			  {
				SET_EXO_ARR(ent, cnt, elt);
				elt->next = NULL;
			  }
		  }
      }
      break;

    case ']':
      exo_err("unexpected ']' encountered");
      break;

    case lex_eof:
      /* nothing to read */
      ent = NULL;
      break;

    default:
      panic("bogus token");
    }

  return ent;
}


int
libexo::yy_nextchar(void)
{
  int c;

  do {
    c = yyinput();
  } while (c == ' ' || c == '\t' || c == '\v' || c == '\f' || c == '\n');
  yyunput(c, yytext);

  return c;
}

int
libexo::yyinput(void)
{
  int c;
  
  *yy_c_buf_p = yy_hold_char;
  
  if ( *yy_c_buf_p == YY_END_OF_BUFFER_CHAR )
	{
	  /* yy_c_buf_p now points to the character we want to return.
	   * If this occurs *before* the EOB characters, then it's a
	   * valid NUL; if not, then we've hit the end of the buffer.
	   */
	  if ( yy_c_buf_p < &yy_current_buffer->yy_ch_buf[yy_n_chars] )
		/* This was really a NUL. */
		*yy_c_buf_p = '\0';
	  
	  else
		{ /* need more input */
		  int offset = yy_c_buf_p - yytext;
		  ++yy_c_buf_p;
		  
		  switch ( yy_get_next_buffer() )
			{
			case EOB_ACT_LAST_MATCH:
			  /* This happens because yy_g_n_b()
			   * sees that we've accumulated a
			   * token and flags that we need to
			   * try matching the token before
			   * proceeding.  But for input(),
			   * there's no matching to consider.
			   * So convert the EOB_ACT_LAST_MATCH
			   * to EOB_ACT_END_OF_FILE.
			   */
			  
			  /* Reset buffer status. */
			  yyrestart( yyin );
			  
			  /* fall through */
			  
			case EOB_ACT_END_OF_FILE:
			  {
				if ( yywrap() )
				  return EOF;
				
				if ( ! yy_did_buffer_switch_on_eof )
				  yyrestart(yyin);
#ifdef __cplusplus
				return yyinput();
#else
				return input();
#endif
			  }
			  
			case EOB_ACT_CONTINUE_SCAN:
			  yy_c_buf_p = yytext + offset;
			  break;
			}
		}
	}
  
  c = *(unsigned char *) yy_c_buf_p;	/* cast for 8-bit char's */
  *yy_c_buf_p = '\0';	/* preserve yytext */
  yy_hold_char = *++yy_c_buf_p;
  
  
  return c;
}


void
libexo::yyunput( int c, register char *yy_bp )
{
  register char *yy_cp = yy_c_buf_p;
  
  /* undo effects of setting up yytext */
  *yy_cp = yy_hold_char;
  
  if ( yy_cp < yy_current_buffer->yy_ch_buf + 2 )
	{ /* need to shift things up to make room */
	  /* +2 for EOB chars. */
	  register int number_to_move = yy_n_chars + 2;
	  register char *dest
		= &yy_current_buffer->yy_ch_buf[yy_current_buffer->yy_buf_size + 2];
	  register char *source =
		&yy_current_buffer->yy_ch_buf[number_to_move];
	  
	  while ( source > yy_current_buffer->yy_ch_buf )
		*--dest = *--source;
	  
	  yy_cp += (int) (dest - source);
	  yy_bp += (int) (dest - source);
	  yy_current_buffer->yy_n_chars =
		yy_n_chars = yy_current_buffer->yy_buf_size;
	  
	  if ( yy_cp < yy_current_buffer->yy_ch_buf + 2 )
		yy_fatal_error( "flex scanner push-back overflow" );
	}
  
  *--yy_cp = (char) c;
  
  
  yytext = yy_bp;
  yy_hold_char = *yy_cp;
  yy_c_buf_p = yy_cp;
}

int
libexo::yy_get_next_buffer()
{
  register char *dest = yy_current_buffer->yy_ch_buf;
  register char *source = yytext;
  register int number_to_move, i;
  int ret_val;

  if ( yy_c_buf_p > &yy_current_buffer->yy_ch_buf[yy_n_chars + 1] )
	yy_fatal_error("fatal flex scanner internal error--end of buffer missed" );

  if ( yy_current_buffer->yy_fill_buffer == 0 )
	{ /* Don't try to fill the buffer, so this is an EOF. */
	  if ( yy_c_buf_p - yytext - YY_MORE_ADJ == 1 )
		{
		  /* We matched a single character, the EOB, so
		   * treat this as a final EOF.
		   */
		  return EOB_ACT_END_OF_FILE;
		}

	  else
		{
		  /* We matched some text prior to the EOB, first
		   * process it.
		   */
		  return EOB_ACT_LAST_MATCH;
		}
	}

  /* Try to read more data. */

  /* First move last chars to start of buffer. */
  number_to_move = (int) (yy_c_buf_p - yytext) - 1;

  for ( i = 0; i < number_to_move; ++i )
	*(dest++) = *(source++);

  if ( yy_current_buffer->yy_buffer_status == YY_BUFFER_EOF_PENDING )
	/* don't do the read, it's not guaranteed to return an EOF,
	 * just force an EOF
	 */
	yy_current_buffer->yy_n_chars = yy_n_chars = 0;

  else
	{
	  int num_to_read =
		yy_current_buffer->yy_buf_size - number_to_move - 1;

	  while ( num_to_read <= 0 )
		{ /* Not enough room in the buffer - grow it. */
#ifdef YY_USES_REJECT
		  yy_fatal_error("input buffer overflow, can't enlarge buffer because scanner uses REJECT" );
#else

		  /* just a shorter name for the current buffer */
		  YY_BUFFER_STATE b = yy_current_buffer;

		  int yy_c_buf_p_offset =
			(int) (yy_c_buf_p - b->yy_ch_buf);

		  if ( b->yy_is_our_buffer )
			{
			  int new_size = b->yy_buf_size * 2;

			  if ( new_size <= 0 )
				b->yy_buf_size += b->yy_buf_size / 8;
			  else
				b->yy_buf_size *= 2;

			  b->yy_ch_buf = (char *)
				/* Include room in for 2 EOB chars. */
				yy_flex_realloc( (void *) b->yy_ch_buf, b->yy_buf_size + 2 );
			}
		  else
			/* Can't grow it, we don't own it. */
			b->yy_ch_buf = 0;

		  if ( ! b->yy_ch_buf )
			yy_fatal_error("fatal error - scanner input buffer overflow" );

		  yy_c_buf_p = &b->yy_ch_buf[yy_c_buf_p_offset];

		  num_to_read = yy_current_buffer->yy_buf_size -
			number_to_move - 1;
#endif
		}

	  if ( num_to_read > YY_READ_BUF_SIZE )
		num_to_read = YY_READ_BUF_SIZE;

	  /* Read in more data. */
	  YY_INPUT( (&yy_current_buffer->yy_ch_buf[number_to_move]),
				yy_n_chars, num_to_read );

	  yy_current_buffer->yy_n_chars = yy_n_chars;
	}

  if ( yy_n_chars == 0 )
	{
	  if ( number_to_move == YY_MORE_ADJ )
		{
		  ret_val = EOB_ACT_END_OF_FILE;
		  yyrestart( yyin );
		}

	  else
		{
		  ret_val = EOB_ACT_LAST_MATCH;
		  yy_current_buffer->yy_buffer_status =
			YY_BUFFER_EOF_PENDING;
		}
	}

  else
	ret_val = EOB_ACT_CONTINUE_SCAN;

  yy_n_chars += number_to_move;
  yy_current_buffer->yy_ch_buf[yy_n_chars] = YY_END_OF_BUFFER_CHAR;
  yy_current_buffer->yy_ch_buf[yy_n_chars + 1] = YY_END_OF_BUFFER_CHAR;

  yytext = &yy_current_buffer->yy_ch_buf[0];

  return ret_val;
}


void
libexo::yyrestart(FILE *input_file)
{
  if ( ! yy_current_buffer )
	yy_current_buffer = yy_create_buffer( yyin, YY_BUF_SIZE );
  
  yy_init_buffer( yy_current_buffer, input_file );
  yy_load_buffer_state();
}

int
libexo::yywrap(void)
{
  return 1;
}

void
libexo::yy_setstream(FILE *stream)
{
  int i;
#define MAX_STREAMS	16
  static struct {
    FILE *stream;
    YY_BUFFER_STATE buffer;
  } streams[MAX_STREAMS] = { {NULL, NULL}, };
  static int num_streams = 0;
  static FILE *last_stream = NULL;

  /* same stream? */
  if (stream == last_stream)
    return;

  /* else, switch to new stream */
  for (i=0; i < num_streams; i++)
    {
      if (streams[i].stream == stream)
		{
		  yy_switch_to_buffer(streams[i].buffer);
		  return;
		}
    }

  /* hrmmm... not found, create a new buffer for this stream */
  if (num_streams == MAX_STREAMS)
    fatal("out of lex buffer streams, increase MAX_STREAMS");

  streams[num_streams].stream = stream;
  streams[num_streams].buffer = yy_create_buffer(stream, YY_BUF_SIZE);
  yy_switch_to_buffer(streams[num_streams].buffer);
  num_streams++;
}

void
libexo::yy_switch_to_buffer( YY_BUFFER_STATE new_buffer )
{
  if ( yy_current_buffer == new_buffer )
	return;

  if ( yy_current_buffer )
	{
	  /* Flush out information for old buffer. */
	  *yy_c_buf_p = yy_hold_char;
	  yy_current_buffer->yy_buf_pos = yy_c_buf_p;
	  yy_current_buffer->yy_n_chars = yy_n_chars;
	}

  yy_current_buffer = new_buffer;
  yy_load_buffer_state();

  /* We don't actually know whether we did this switch during
   * EOF (yywrap()) processing, but the only time this flag
   * is looked at is after yywrap() is called, so it's safe
   * to go ahead and always set it.
   */
  yy_did_buffer_switch_on_eof = 1;
}


int
libexo::yylex(void)
{
  register yy_state_type yy_current_state;
  register char *yy_cp, *yy_bp;
  register int yy_act;
  
  //#line 87 "exolex.l"
  
  
  //#line 655 "lex.yy.c"
  
  if ( yy_init ) {
	yy_init = 0;
	
#ifdef YY_USER_INIT
	YY_USER_INIT;
#endif
	
	if ( ! yy_start )
	  yy_start = 1;	/* first start state */
	
	if ( ! yyin )
	  yyin = stdin;
	
	  if ( ! yyout )
		yyout = stdout;

	  if ( ! yy_current_buffer )
		yy_current_buffer =
		  yy_create_buffer( yyin, YY_BUF_SIZE );

	  yy_load_buffer_state();
	}

  while ( 1 ) { /* loops until end-of-file is reached */
	yy_cp = yy_c_buf_p;
	
	/* Support of yytext. */
	*yy_cp = yy_hold_char;
	
	/* yy_bp points to the position in yy_ch_buf of the start of
	   * the current run.
	   */
	  yy_bp = yy_cp;

	  yy_current_state = yy_start;
  yy_match:
	  do{
		register YY_CHAR yy_c = yy_ec[YY_SC_TO_UI(*yy_cp)];
		if ( yy_accept[yy_current_state] ) {
		  yy_last_accepting_state = yy_current_state;
		  yy_last_accepting_cpos = yy_cp;
		}
		while ( yy_chk[yy_base[yy_current_state] + yy_c] != yy_current_state){
		  yy_current_state = (int) yy_def[yy_current_state];
		  if ( yy_current_state >= 60 )
			yy_c = yy_meta[(unsigned int) yy_c];
		}
		yy_current_state = yy_nxt[yy_base[yy_current_state] + (unsigned int) yy_c];
		++yy_cp;
	  }
	  while ( yy_base[yy_current_state] != 178 );
	  
  yy_find_action:
	  yy_act = yy_accept[yy_current_state];
	  if ( yy_act == 0 )
		{ /* have to back up */
		  yy_cp = yy_last_accepting_cpos;
		  yy_current_state = yy_last_accepting_state;
		  yy_act = yy_accept[yy_current_state];
		}
	  
	  YY_DO_BEFORE_ACTION;
	  
  do_action:	/* This label is used only to access EOF actions. */

	  switch ( yy_act )
		{ /* beginning of action switch */
		case 0: /* must back up */
		  /* undo the effects of YY_DO_BEFORE_ACTION */
		  *yy_cp = yy_hold_char;
		  yy_cp = yy_last_accepting_cpos;
		  yy_current_state = yy_last_accepting_state;
		  goto yy_find_action;
		  
		case 1:
		  YY_RULE_SETUP
			//#line 89 "exolex.l"
			{ return lex_byte; }
		  break;
		case 2:
		  YY_RULE_SETUP
			//#line 91 "exolex.l"
			{ lex_eat_comment(); }
		  break;
		case 3:
		  YY_RULE_SETUP
			//#line 93 "exolex.l"
			{ return lex_address; }
		  break;
		case 4:
		  YY_RULE_SETUP
			//#line 94 "exolex.l"
			{ return lex_integer; }
		  break;
		case 5:
		  YY_RULE_SETUP
			//#line 95 "exolex.l"
			{ return lex_integer; }
		  break;
		case 6:
		  YY_RULE_SETUP
			//#line 97 "exolex.l"
			{ return lex_float; }
		  break;
		case 7:
		  YY_RULE_SETUP
			//#line 98 "exolex.l"
			{ return lex_float; }
		  break;
		case 8:
		  YY_RULE_SETUP
			//#line 99 "exolex.l"
			{ return lex_float; }
		  break;
		case 9:
		  YY_RULE_SETUP
			//#line 101 "exolex.l"
			{ return lex_char; }
		  break;
		case 10:
		  YY_RULE_SETUP
			//#line 103 "exolex.l"
			/**/	{ return lex_string; }
		  break;
		case 11:
		  YY_RULE_SETUP
			//#line 105 "exolex.l"
			{ return lex_token; }
		  break;
		case 12:
		  YY_RULE_SETUP
			//#line 107 "exolex.l"
			{ return lex_token; }
		  break;
		case 13:
		  YY_RULE_SETUP
			//#line 109 "exolex.l"
			{ return '{'; }
		  break;
		case 14:
		  YY_RULE_SETUP
			//#line 110 "exolex.l"
			{ return '}'; }
		  break;
		case 15:
		  YY_RULE_SETUP
			//#line 111 "exolex.l"
			{ return ','; }
		  break;
		case 16:
		  YY_RULE_SETUP
			//#line 112 "exolex.l"
			{ return '('; }
		  break;
		case 17:
		  YY_RULE_SETUP
			//#line 113 "exolex.l"
			{ return ')'; }
		  break;
		case 18:
		  YY_RULE_SETUP
			//#line 114 "exolex.l"
			{ return '['; }
		  break;
		case 19:
		  YY_RULE_SETUP
			//#line 115 "exolex.l"
			{ return ']'; }
		  break;
		case 20:
		  YY_RULE_SETUP
			//#line 116 "exolex.l"
			{ BEGIN(BLOB_MODE); return '<'; }
		  break;
		case 21:
		  YY_RULE_SETUP
			//#line 117 "exolex.l"
			{ BEGIN(INITIAL); return '>'; }
		  break;
		case 22:
		  YY_RULE_SETUP
			//#line 119 "exolex.l"
			{ /* nada */; }
		  break;
		case 23:
		  YY_RULE_SETUP
			//#line 120 "exolex.l"
			{ line++; }
		  break;
		case YY_STATE_EOF(INITIAL):
		case YY_STATE_EOF(BLOB_MODE):
		  //#line 121 "exolex.l"
		  { return lex_eof; }
		  break;
		case 24:
		  YY_RULE_SETUP
			//#line 122 "exolex.l"
			{ /* bogus char */
			  fatal("bogus character in input"); }
		  break;
		case 25:
		  YY_RULE_SETUP
			//#line 125 "exolex.l"
			fwrite(yytext, yyleng, 1, yyout);
		  break;
		  //#line 869 "lex.yy.c"

		case YY_END_OF_BUFFER:
		  {
			/* Amount of text matched not including the EOB char. */
			int yy_amount_of_matched_text = (int) (yy_cp - yytext) - 1;

			/* Undo the effects of YY_DO_BEFORE_ACTION. */
			*yy_cp = yy_hold_char;
			YY_RESTORE_YY_MORE_OFFSET

			  if ( yy_current_buffer->yy_buffer_status == YY_BUFFER_NEW ) {
				  /* We're scanning a new file or input source.  It's
				   * possible that this happened because the user
				   * just pointed yyin at a new source and called
				   * yylex().  If so, then we have to assure
				   * consistency between yy_current_buffer and our
				   * globals.  Here is the right place to do so, because
				   * this is the first action (other than possibly a
				   * back-up) that will match for the new input source.
				   */
				  yy_n_chars = yy_current_buffer->yy_n_chars;
				  yy_current_buffer->yy_input_file = yyin;
				  yy_current_buffer->yy_buffer_status = YY_BUFFER_NORMAL;
			  }

			/* Note that here we test for yy_c_buf_p <= to the position
			 * of the first EOB in the buffer, since yy_c_buf_p will
			 * already have been incremented past the NUL character
			 * (since all states make transitions on EOB to the
			 * end-of-buffer state).  Contrast this with the test
			 * in input().
			 */
			if ( yy_c_buf_p <= &yy_current_buffer->yy_ch_buf[yy_n_chars] ) {
			  /* This was really a NUL. */
			  yy_state_type yy_next_state;
			  
				yy_c_buf_p = yytext + yy_amount_of_matched_text;
				
				yy_current_state = yy_get_previous_state();

				/* Okay, we're now positioned to make the NUL
				 * transition.  We couldn't have
				 * yy_get_previous_state() go ahead and do it
				 * for us because it doesn't know how to deal
				 * with the possibility of jamming (and we don't
				 * want to build jamming into it because then it
				 * will run more slowly).
				 */

				yy_next_state = yy_try_NUL_trans( yy_current_state );

				yy_bp = yytext + YY_MORE_ADJ;

				if ( yy_next_state ) {
					/* Consume the NUL. */
					yy_cp = ++yy_c_buf_p;
					yy_current_state = yy_next_state;
					goto yy_match;
				  } else {
					yy_cp = yy_c_buf_p;
					goto yy_find_action;
				  }
			} else switch ( yy_get_next_buffer() ) {
			case EOB_ACT_END_OF_FILE:
			  {
				yy_did_buffer_switch_on_eof = 0;
				
				if ( yywrap() ) {
				  /* Note: because we've taken care in
				   * yy_get_next_buffer() to have set up
				   * yytext, we can now set up
				   * yy_c_buf_p so that if some total
					   * hoser (like flex itself) wants to
					   * call the scanner after we return the
					   * YY_NULL, it'll still work - another
					   * YY_NULL will get returned.
					   */
					  yy_c_buf_p = yytext + YY_MORE_ADJ;

					  yy_act = YY_STATE_EOF(YY_START);
					  goto do_action;
				} else {
				  if ( ! yy_did_buffer_switch_on_eof )
					yyrestart(yyin);
				}
				break;
			  }
			  
			case EOB_ACT_CONTINUE_SCAN:
			  yy_c_buf_p =
				yytext + yy_amount_of_matched_text;
			  
			  yy_current_state = yy_get_previous_state();

				yy_cp = yy_c_buf_p;
				yy_bp = yytext + YY_MORE_ADJ;
				goto yy_match;

			  case EOB_ACT_LAST_MATCH:
				yy_c_buf_p =
				  &yy_current_buffer->yy_ch_buf[yy_n_chars];

				yy_current_state = yy_get_previous_state();

				yy_cp = yy_c_buf_p;
				yy_bp = yytext + YY_MORE_ADJ;
				goto yy_find_action;
			}
			break;
		  }

		default:
		  yy_fatal_error("fatal flex scanner internal error--no action found");
		} /* end of action switch */
	} /* end of scanning one token */
} /* end of yylex */

int
libexo::yy_get_previous_state()
{
  register yy_state_type yy_current_state;
  register char *yy_cp;
  
  yy_current_state = yy_start;
  
  for ( yy_cp = yytext + YY_MORE_ADJ; yy_cp < yy_c_buf_p; ++yy_cp )
	{
	  register YY_CHAR yy_c = (*yy_cp ? yy_ec[YY_SC_TO_UI(*yy_cp)] : 1);
	  if ( yy_accept[yy_current_state] )
		{
		  yy_last_accepting_state = yy_current_state;
		  yy_last_accepting_cpos = yy_cp;
		}
	  while ( yy_chk[yy_base[yy_current_state] + yy_c] != yy_current_state )
		{
		  yy_current_state = (int) yy_def[yy_current_state];
		  if ( yy_current_state >= 60 )
			yy_c = yy_meta[(unsigned int) yy_c];
		}
	  yy_current_state = yy_nxt[yy_base[yy_current_state] + (unsigned int) yy_c];
	}

  return yy_current_state;
}

void
libexo::yy_fatal_error( const char msg[] )
{
  fprintf( stderr, "%s\n", msg );
  exit( YY_EXIT_FAILURE );
}

libexo::YY_BUFFER_STATE
libexo::yy_create_buffer( FILE *file, int size )
{
  YY_BUFFER_STATE b;
  
  b = (YY_BUFFER_STATE) yy_flex_alloc( sizeof( struct yy_buffer_state ) );
  if ( ! b )
	yy_fatal_error( "out of dynamic memory in yy_create_buffer()" );

  b->yy_buf_size = size;

  /* yy_ch_buf has to be 2 characters longer than the size given because
   * we need to put in 2 end-of-buffer characters.
   */
  b->yy_ch_buf = (char *) yy_flex_alloc( b->yy_buf_size + 2 );
  if ( ! b->yy_ch_buf )
	yy_fatal_error( "out of dynamic memory in yy_create_buffer()" );

  b->yy_is_our_buffer = 1;

  yy_init_buffer( b, file );

  return b;
}

void
libexo::yy_init_buffer( YY_BUFFER_STATE b, FILE *file )
{
  yy_flush_buffer( b );

  b->yy_input_file = file;
  b->yy_fill_buffer = 1;

#if YY_ALWAYS_INTERACTIVE
  b->yy_is_interactive = 1;
#else
#if YY_NEVER_INTERACTIVE
  b->yy_is_interactive = 0;
#else
  b->yy_is_interactive = file ? (isatty( fileno(file) ) > 0) : 0;
#endif
#endif
}

void
libexo::yy_flush_buffer( YY_BUFFER_STATE b )
{
  if ( ! b )
	return;

  b->yy_n_chars = 0;

  /* We always need two end-of-buffer characters.  The first causes
   * a transition to the end-of-buffer state.  The second causes
   * a jam in that state.
   */
  b->yy_ch_buf[0] = YY_END_OF_BUFFER_CHAR;
  b->yy_ch_buf[1] = YY_END_OF_BUFFER_CHAR;

  b->yy_buf_pos = &b->yy_ch_buf[0];

  b->yy_at_bol = 1;
  b->yy_buffer_status = YY_BUFFER_NEW;

  if ( b == yy_current_buffer )
	yy_load_buffer_state();
}

void
libexo::yy_load_buffer_state( void )
{
  yy_n_chars = yy_current_buffer->yy_n_chars;
  yytext = yy_c_buf_p = yy_current_buffer->yy_buf_pos;
  yyin = yy_current_buffer->yy_input_file;
  yy_hold_char = *yy_c_buf_p;
}

void
libexo::lex_eat_comment(void)
{
  char c, c1;

 loop:
  while ((c = yyinput()) != '*' && c != 0)
    {
      if (c == '\n')
		line++;
    }

  if ((c1 = yyinput()) != '/' && c1 != 0)
    {
      yyunput(c1, yytext);
      goto loop;
    }
}

int
libexo::yy_try_NUL_trans( yy_state_type yy_current_state )
{
  register int yy_is_jam;
  register char *yy_cp = yy_c_buf_p;
  
  register YY_CHAR yy_c = 1;
  if ( yy_accept[yy_current_state] )
	{
	  yy_last_accepting_state = yy_current_state;
	  yy_last_accepting_cpos = yy_cp;
	}
  while ( yy_chk[yy_base[yy_current_state] + yy_c] != yy_current_state )
	{
	  yy_current_state = (int) yy_def[yy_current_state];
	  if ( yy_current_state >= 60 )
		yy_c = yy_meta[(unsigned int) yy_c];
	}
  yy_current_state = yy_nxt[yy_base[yy_current_state] + (unsigned int) yy_c];
  yy_is_jam = (yy_current_state == 59);

  return yy_is_jam ? 0 : yy_current_state;
}
