/* memory.h - flat memory space interfaces */

/* SimpleScalar(TM) Tool Suite
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 * All Rights Reserved. 
 * 
 * THIS IS A LEGAL DOCUMENT, BY USING SIMPLESCALAR,
 * YOU ARE AGREEING TO THESE TERMS AND CONDITIONS.
 * 
 * No portion of this work may be used by any commercial entity, or for any
 * commercial purpose, without the prior, written permission of SimpleScalar,
 * LLC (info@simplescalar.com). Nonprofit and noncommercial use is permitted
 * as described below.
 * 
 * 1. SimpleScalar is provided AS IS, with no warranty of any kind, express
 * or implied. The user of the program accepts full responsibility for the
 * application of the program and the use of any results.
 * 
 * 2. Nonprofit and noncommercial use is encouraged. SimpleScalar may be
 * downloaded, compiled, executed, copied, and modified solely for nonprofit,
 * educational, noncommercial research, and noncommercial scholarship
 * purposes provided that this notice in its entirety accompanies all copies.
 * Copies of the modified software can be delivered to persons who use it
 * solely for nonprofit, educational, noncommercial research, and
 * noncommercial scholarship purposes provided that this notice in its
 * entirety accompanies all copies.
 * 
 * 3. ALL COMMERCIAL USE, AND ALL USE BY FOR PROFIT ENTITIES, IS EXPRESSLY
 * PROHIBITED WITHOUT A LICENSE FROM SIMPLESCALAR, LLC (info@simplescalar.com).
 * 
 * 4. No nonprofit user may place any restrictions on the use of this software,
 * including as modified by the user, by any other authorized user.
 * 
 * 5. Noncommercial and nonprofit users may distribute copies of SimpleScalar
 * in compiled or executable form as set forth in Section 2, provided that
 * either: (A) it is accompanied by the corresponding machine-readable source
 * code, or (B) it is accompanied by a written offer, with no time limit, to
 * give anyone a machine-readable copy of the corresponding source code in
 * return for reimbursement of the cost of distribution. This written offer
 * must permit verbatim duplication by anyone, or (C) it is distributed by
 * someone who received only the executable form, and is accompanied by a
 * copy of the written offer of source code.
 * 
 * 6. SimpleScalar was developed by Todd M. Austin, Ph.D. The tool suite is
 * currently maintained by SimpleScalar LLC (info@simplescalar.com). US Mail:
 * 2395 Timbercrest Court, Ann Arbor, MI 48105.
 * 
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 */


#ifndef MEMORY_SIMPLE_H
#define MEMORY_SIMPLE_H

#include <stdio.h>
#include <stdlib.h>
#include <isis/host.h>
#include <isis/misc.h>
#include <isis/machine.h>
#include <isis/stats.h>

#define MEM_PTAB_SIZE		(32*1024)
#define MEM_LOG_PTAB_SIZE	15

/* page table entry */
struct mem_pte_t {
  struct mem_pte_t *next;	/* next translation in this bucket */
  md_addr_t tag;		/* virtual page number tag */
  byte_t *page;			/* page pointer */
};

/* memory object */
struct mem_t {
  /* memory object state */
  char *name;				/* name of this memory space */
  struct mem_pte_t *ptab[MEM_PTAB_SIZE];/* inverted page table */
  /* memory object stats */
  counter_t page_count;			/* total number of pages allocated */
  counter_t ptab_misses;		/* total first level page tbl misses */
  counter_t ptab_accesses;		/* total page table accesses */
};

/* memory access command */
enum mem_cmd {
  Read,                 /* read memory from target (simulated prog) to host */
  Write                 /* write memory from host (simulator) to target */
};

#define MEM_PTAB_SET(ADDR) (((ADDR) >> MD_LOG_PAGE_SIZE) & (MEM_PTAB_SIZE - 1))
#define MEM_PTAB_TAG(ADDR)  ((ADDR) >> (MD_LOG_PAGE_SIZE + MEM_LOG_PTAB_SIZE))
#define MEM_OFFSET(ADDR)	((ADDR) & (MD_PAGE_SIZE - 1))

class memory_simple
{
 private:
  typedef stats_simple stats_simple_type;
  stats_simple_type *st_simple;

 public:
  memory_simple(){st_simple = NULL;}
  void connect_stats(stats_simple_type& a){ st_simple = &a; }
  byte_t * mem_translate(struct mem_t *mem,	md_addr_t addr);  
  void mem_newpage(struct mem_t *mem, md_addr_t addr);
  byte_t mem_read_byte(struct mem_t *mem, md_addr_t addr);
  word_t mem_read_word(struct mem_t *mem, md_addr_t addr);
  void mem_write_byte(struct mem_t *mem, md_addr_t addr, byte_t val);
  byte_t * mem_page(struct mem_t *mem, md_addr_t addr);
  void mem_tickle(struct mem_t *mem, md_addr_t addr);
  /* create a flat memory space */
  struct mem_t * mem_create(char *name);	/* name of the memory space */
  /* generic memory access function, it's safe because alignments and 
	 permissions are checked, handles any natural transfer sizes; 
	 note, faults out if nbytes is not a power-of-two or larger 
	 then MD_PAGE_SIZE */
  enum md_fault_type 
	mem_access(struct mem_t *mem, /* memory space to access */
			   enum mem_cmd cmd,		/* Read (from sim mem) or Write */
			   md_addr_t addr,		/* target address to access */
			   void *vp,			/* host memory address to access */
			   int nbytes,			/* number of bytes to access */  
			   bool flag);
  /* register memory system-specific statistics */
  void mem_reg_stats(struct mem_t *mem,	/* memory space to declare */
					 struct stat_sdb_t *sdb);	/* stats data base */
  
  /* initialize memory system, call before loader.c */
  void mem_init(struct mem_t *mem);	/* memory space to initialize */
  
  /* dump a block of memory, returns any faults encountered */
  enum md_fault_type mem_dump(struct mem_t *mem, /* memory space to display */
							  md_addr_t addr,	/* target address to dump */
							  int len,		 /* number bytes to dump */
							  FILE *stream);	  /* output stream */  
  /*
   * memory accessor routines, these routines require a memory access function
   * definition to access memory, the memory access function provides a "hook"
   * for programs to instrument memory accesses, this is used by various
   * simulators for various reasons; for the default operation - direct access
   * to the memory system, pass mem_access() as the memory access function
   */
  /* copy a '\0' terminated string to/from simulated memory space, returns
	 the number of bytes copied, returns any fault encountered */
  enum md_fault_type
	mem_strcpy(struct mem_t *mem,		/* memory space to access */
			   enum mem_cmd cmd,		/* Read (from sim mem) or Write */
			   md_addr_t addr,		/* target address to access */
			   char *s);			/* host memory string buffer */

  /* copy NBYTES to/from simulated memory space, returns any faults */
  enum md_fault_type
	mem_bcopy(struct mem_t *mem,		/* memory space to access */
			  enum mem_cmd cmd,		/* Read (from sim mem) or Write */
			  md_addr_t addr,		/* target address to access */
			  void *vp,			/* host memory address to access */
			  int nbytes);			/* number of bytes to access */
  /* copy NBYTES to/from simulated memory space, NBYTES must be a multiple
	 of 4 bytes, this function is faster than mem_bcopy(), returns any
	 faults encountered */
  enum md_fault_type
	mem_bcopy4(struct mem_t *mem,		/* memory space to access */
			   enum mem_cmd cmd,		/* Read (from sim mem) or Write */
			   md_addr_t addr,		/* target address to access */
			   void *vp,			/* host memory address to access */
			   int nbytes);			/* number of bytes to access */
  
  /* zero out NBYTES of simulated memory, returns any faults encountered */
  enum md_fault_type
	mem_bzero(struct mem_t *mem,		/* memory space to access */
			  md_addr_t addr,		/* target address to access */
			  int nbytes);			/* number of bytes to clear */
};

#endif /* MEMORY_H */
