/*
 * <<< memory_test.cc >>>
 *
 * --- Test program for memory class
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif // HAVE_CONFIG_H
#include <iostream>
#include "memory.h"

#define rnd() ((unsigned int)((seed = 1566083941UL * seed + 1) >> 16))
static unsigned long seed = 1;

#if (SIZEOF_INT >= 4)
 typedef unsigned int word;
#else // !(SIZEOF_INT >= 4)
 typedef unsigned long word;
#endif // !(SIZEOF_INT >= 4)
const size_t sizeof_word = 4;

int main(void)
{
	const word size = 0x20;
	memory<word, word, sizeof_word> mem(size * sizeof_word);
	{
		// initial state
		cout << mem << endl;
	}
	{
		// word access test
		mem.write(0, 1);
		mem.write(sizeof_word, 1);
		for (size_t i = 2; i < size; i++) {
			word adr = i * sizeof_word;
			word adr_prev1 = adr - sizeof_word;
			word adr_prev2 = adr_prev1 - sizeof_word;
			mem.write(adr, mem.read(adr_prev1) + mem.read(adr_prev2));
		}
		cout << mem << endl;
	}
	{
		// byte access test for big-endian
		mem.set_big_endian();
		word i;
		for (i = 0; i < mem.size(); i++) {
			mem.write_char(i, char(i));
		}
		for (i = 0; i < mem.size(); i++) {
			if (mem.read_char(i) != char(i)) return 1;
		}
		cout << mem << endl;
	}
	{
		// byte access test for little-endian
		mem.set_little_endian();
		word i;
		for (i = 0; i < mem.size(); i++) {
			mem.write_char(i, char(i));
		}
		for (i = 0; i < mem.size(); i++) {
			if (mem.read_char(i) != char(i)) return 1;
		}
		cout << mem << endl;
	}
	{
		// resize test
		mem.resize(mem.size() + 0x10);
		cout << mem << endl;
	}
	{
		// random access test
		const size_t count = 16 * 1024; // 16k times
		const word size = 256 * 1024; // 256k bytes
		mem.clear();
		mem.resize(size);
		word* table = new word[size / sizeof_word];
		size_t i;
		for (i = 0; i < size / sizeof_word; i++) table[i] = 0;
		for (i = 0; i < count; i++) {
			word address = (((rnd() >> 2) & 0x3fff) << 2);
			word data = (rnd() & 0xffff);
			mem.write(address, data);
			table[address / sizeof_word] = data;
		}
		for (i = 0; i < size / sizeof_word; i++) {
			word address = i * sizeof_word, data = table[i];
			if (mem.read(address) != data) {
				cerr << hex << "*0x" << address << " = 0x" << mem.read(address)
					 << " (correct: 0x" << data << ')' << endl;
				return 1;
			}
		}
		delete[] table;
	}
	return 0;
}
