/* -*- C++ -*-
 *
 * <<< write_buffer.h >>>
 *
 * --- Write buffer class 'write_buffer'
 *     Copyright (C) 1995-1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef WRITE_BUFFER_H
#define WRITE_BUFFER_H 1

#include <iostream>
#include <utility>
#include "root_object.h"
#include "cyclic_queue.h"

template < class A, class D = A, class C = cyclic_queue< pair<A, D> > >
class write_buffer : public root_object
{
public:
	typedef A address_type;
	typedef D data_type;
	typedef C container_type;
	typedef typename C::size_type size_type;
private:
	typedef write_buffer<A, D, C> thisclass;
	typedef root_object inherited;
	container_type buf;
public:
	explicit write_buffer(size_type a = 0);
	write_buffer(const thisclass&);
	virtual ~write_buffer();
	virtual void output(ostream&) const;
	size_type size(void) const { return buf.max_size(); }
	void resize(size_type a) { buf.reserve(a); }
	size_type length(void) const { return buf.size(); }
	bool empty(void) const { return buf.empty(); }
	bool full(void) const { return buf.full(); }
	const address_type& address(void) const { return buf.front().first; }
	address_type& address(void) { return buf.front().first; }
	const data_type& data(void) const { return buf.front().second; }
	data_type& data(void) { return buf.front().second; }
	void push(const address_type& a, const data_type& b)
		{ buf.push(pair<address_type, data_type>(a, b)); }
	inline bool buffered(const address_type&) const;
	inline data_type read(const address_type&) const;
	void clear(void);
	void pop(void) { buf.pop(); }
};

template <class A, class D, class C>
write_buffer<A, D, C>::write_buffer(write_buffer<A, D, C>::size_type a)
	: buf(a)
{}

template <class A, class D, class C>
write_buffer<A, D, C>::write_buffer(const write_buffer<A, D, C>& a)
	: buf(a.buf)
{}

template <class A, class D, class C>
write_buffer<A, D, C>::~write_buffer()
{}

template <class A, class D, class C>
inline bool write_buffer<A, D, C>::buffered
	(const write_buffer<A, D, C>::address_type& a) const
{
	for (size_type i = 0; i < length(); i++) {
		if (buf[i].first == a) return true;
	}
	return false;
}

template <class A, class D, class C>
inline write_buffer<A, D, C>::data_type write_buffer<A, D, C>::read
	(const write_buffer<A, D, C>::address_type& a) const
{
	for (size_type i = length() - 1; ; i--) {
		if (buf[i].first == a) return buf[i].second;
		if (i == 0) break;
	}
	return data_type(0);
}

template <class A, class D, class C>
void write_buffer<A, D, C>::clear(void)
{
	buf.clear();
}

template <class A, class D, class C>
void write_buffer<A, D, C>::output(ostream& os) const
{
	if (empty()) return;
	const long flags = os.flags();
	for (size_type i = 0; i < length(); i++) {
		os << "*0x" << hex << buf[i].first << "=0x" << buf[i].second;
		if (i < length() - 1) os << ' ';
	}
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* WRITE_BUFFER_H */
