/* -*- C++ -*-
 *
 * <<< bus_packet.h >>>
 *
 * --- Bus packet class 'bus_packet'
 *     Copyright (C) 1997-1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef BUS_PACKET_H
#define BUS_PACKET_H 1

#ifdef DEBUG
# include <cassert>
#endif // DEBUG
#include <iostream>
#include "bus_packet_base.h"

template <class A, class D = A>
class bus_packet : public bus_packet_base<A, D>
{
private:
	typedef bus_packet<A, D> thisclass;
	typedef bus_packet_base<A, D> inherited;
public:
	typedef typename inherited::address_type address_type;
	typedef typename inherited::data_type data_type;
private:
	static const int rw_mask = 0x10;
	static const int type_mask = 0x7;
	static const int read_pattern = 0x0;
	static const int write_pattern = 0x10;
	static const int req_pattern = 0x0;
	static const int grant_pattern = 0x1;
	static const int ack_pattern = 0x2;
	static const int nack_pattern = 0x3;
	static const int data_pattern = 0x4;
	int flags_;
public:
	bus_packet(void) : flags_(0) {}
	virtual ~bus_packet() {}
	virtual root_object* new_object(void) const
		{ return new thisclass; }
	virtual root_object* clone_object(void) const
		{ return new thisclass(*this); }
	virtual void input(istream&);
	virtual void output(ostream&) const;
	// basic virtual functions
	virtual bool is_read(void) const
		{ return (flags_ & rw_mask) == read_pattern; }
	virtual bool is_write(void) const
		{ return (flags_ & rw_mask) == write_pattern; }
	virtual bool is_request(void) const
		{ return (flags_ & type_mask) == req_pattern; }
	virtual bool is_grant(void) const
		{ return (flags_ & type_mask) == grant_pattern; }
	virtual bool is_ack(void) const
		{ return (flags_ & type_mask) == ack_pattern; }
	virtual bool is_nack(void) const
		{ return (flags_ & type_mask) == nack_pattern; }
	virtual bool is_data(void) const
		{ return (flags_ & type_mask) == data_pattern; }
	virtual void set_read(void)
		{ flags_ = (flags_ & ~rw_mask) | read_pattern; }
	virtual void set_write(void)
		{ flags_ = (flags_ & ~rw_mask) | write_pattern; }
	virtual void set_request(void)
		{ flags_ = (flags_ & ~type_mask) | req_pattern; }
	virtual void set_ack(void)
		{ flags_ = (flags_ & ~type_mask) | ack_pattern; }
	virtual void set_grant(void)
		{ flags_ = (flags_ & ~type_mask) | grant_pattern; }
	virtual void set_nack(void)
		{ flags_ = (flags_ & ~type_mask) | nack_pattern; }
	virtual void set_data(void)
		{ flags_ = (flags_ & ~type_mask) | data_pattern; }
	// useful virtual functions
	// is_{read,write}_{request,grant,ack,nack,data}()
	virtual bool is_read_request(void) const
		{ return flags_ == (read_pattern | req_pattern); }
	virtual bool is_read_grant(void) const
		{ return flags_ == (read_pattern | grant_pattern); }
	virtual bool is_read_ack(void) const
		{ return flags_ == (read_pattern | ack_pattern); }
	virtual bool is_read_nack(void) const
		{ return flags_ == (read_pattern | nack_pattern); }
	virtual bool is_read_data(void) const
		{ return flags_ == (read_pattern | data_pattern); }
	virtual bool is_write_request(void) const
		{ return flags_ == (write_pattern | req_pattern); }
	virtual bool is_write_grant(void) const
		{ return flags_ == (write_pattern | grant_pattern); }
	virtual bool is_write_ack(void) const
		{ return flags_ == (write_pattern | ack_pattern); }
	virtual bool is_write_nack(void) const
		{ return flags_ == (write_pattern | nack_pattern); }
	virtual bool is_write_data(void) const
		{ return flags_ == (write_pattern | data_pattern); }
	// is_single_{read,write}_{request,grant,ack,nack,data}()
	virtual bool is_single_read_request(void) const
		{ return is_single() && thisclass::is_read_request(); }
	virtual bool is_single_read_grant(void) const
		{ return is_single() && thisclass::is_read_grant(); }
	virtual bool is_single_read_ack(void) const
		{ return is_single() && thisclass::is_read_ack(); }
	virtual bool is_single_read_nack(void) const
		{ return is_single() && thisclass::is_read_nack(); }
	virtual bool is_single_read_data(void) const
		{ return is_single() && thisclass::is_read_data(); }
	virtual bool is_single_write_request(void) const
		{ return is_single() && thisclass::is_write_request(); }
	virtual bool is_single_write_grant(void) const
		{ return is_single() && thisclass::is_write_grant(); }
	virtual bool is_single_write_ack(void) const
		{ return is_single() && thisclass::is_write_ack(); }
	virtual bool is_single_write_nack(void) const
		{ return is_single() && thisclass::is_write_nack(); }
	virtual bool is_single_write_data(void) const
		{ return is_single() && thisclass::is_write_data(); }
	// is_multi_{read,write}_{request,grant,ack,nack,data}()
	virtual bool is_multi_read_request(void) const
		{ return is_multi() && thisclass::is_read_request(); }
	virtual bool is_multi_read_grant(void) const
		{ return is_multi() && thisclass::is_read_grant(); }
	virtual bool is_multi_read_ack(void) const
		{ return is_multi() && thisclass::is_read_ack(); }
	virtual bool is_multi_read_nack(void) const
		{ return is_multi() && thisclass::is_read_nack(); }
	virtual bool is_multi_read_data(void) const
		{ return is_multi() && thisclass::is_read_data(); }
	virtual bool is_multi_write_request(void) const
		{ return is_multi() && thisclass::is_write_request(); }
	virtual bool is_multi_write_grant(void) const
		{ return is_multi() && thisclass::is_write_grant(); }
	virtual bool is_multi_write_ack(void) const
		{ return is_multi() && thisclass::is_write_ack(); }
	virtual bool is_multi_write_nack(void) const
		{ return is_multi() && thisclass::is_write_nack(); }
	virtual bool is_multi_write_data(void) const
		{ return is_multi() && thisclass::is_write_data(); }
	// set_{read,write}_{request,grant,ack,nack,data}()
	virtual void set_read_request(void)
		{ flags_ = (read_pattern | req_pattern); }
	virtual void set_read_request(const address_type& a)
		{ thisclass::set_read_request(), address() = a; }
	virtual void set_read_grant(void)
		{ flags_ = (read_pattern | grant_pattern); }
	virtual void set_read_ack(void)
		{ flags_ = (read_pattern | ack_pattern); }
	virtual void set_read_nack(void)
		{ flags_ = (read_pattern | nack_pattern); }
	virtual void set_read_data(void)
		{ flags_ = (read_pattern | data_pattern); }
	virtual void set_read_data(const data_type& a)
		{ thisclass::set_read_data(), data() = a; }
	virtual void set_write_request(void)
		{ flags_ = (write_pattern | req_pattern); }
	virtual void set_write_request(const address_type& a)
		{ thisclass::set_write_request(), address() = a; }
	virtual void set_write_grant(void)
		{ flags_ = (write_pattern | grant_pattern); }
	virtual void set_write_ack(void)
		{ flags_ = (write_pattern | ack_pattern); }
	virtual void set_write_nack(void)
		{ flags_ = (write_pattern | nack_pattern); }
	virtual void set_write_data(void)
		{ flags_ = (write_pattern | data_pattern); }
	virtual void set_write_data(const data_type& a)
		{ thisclass::set_write_data(), data() = a; }
	// set_single_{read,write}_{request,grant,ack,nack,data}()
	virtual void set_single_read_request(void)
		{ set_single(), thisclass::set_read_request(); }
	virtual void set_single_read_request(const address_type& a)
		{ set_single(), thisclass::set_read_request(a); }
	virtual void set_single_read_grant(void)
		{ set_single(), thisclass::set_read_grant(); }
	virtual void set_single_read_ack(void)
		{ set_single(), thisclass::set_read_ack(); }
	virtual void set_single_read_nack(void)
		{ set_single(), thisclass::set_read_nack(); }
	virtual void set_single_read_data(void)
		{ set_single(), thisclass::set_read_data(); }
	virtual void set_single_read_data(const data_type& a)
		{ set_single(), thisclass::set_read_data(a); }
	virtual void set_single_write_request(void)
		{ set_single(), thisclass::set_write_request(); }
	virtual void set_single_write_request(const address_type& a)
		{ set_single(), thisclass::set_write_request(a); }
	virtual void set_single_write_grant(void)
		{ set_single(), thisclass::set_write_grant(); }
	virtual void set_single_write_ack(void)
		{ set_single(), thisclass::set_write_ack(); }
	virtual void set_single_write_nack(void)
		{ set_single(), thisclass::set_write_nack(); }
	virtual void set_single_write_data(void)
		{ set_single(), thisclass::set_write_data(); }
	virtual void set_single_write_data(const data_type& a)
		{ set_single(), thisclass::set_write_data(a); }
	// set_multi_{read,write}_{request,grant,ack,nack,data}()
	virtual void set_multi_read_request(void)
		{ set_multi(), thisclass::set_read_request(); }
	virtual void set_multi_read_request(const address_type& a, size_t b)
		{ set_multi(b), thisclass::set_read_request(a); }
	virtual void set_multi_read_grant(void)
		{ set_multi(), thisclass::set_read_grant(); }
	virtual void set_multi_read_ack(void)
		{ set_multi(), thisclass::set_read_ack(); }
	virtual void set_multi_read_nack(void)
		{ set_multi(), thisclass::set_read_nack(); }
	virtual void set_multi_read_data(void)
		{ set_multi(), thisclass::set_read_data(); }
	virtual void set_multi_read_data(const data_type& a, size_t b, size_t c)
		{ set_multi(b, c), thisclass::set_read_data(a); }
	virtual void set_multi_write_request(void)
		{ set_multi(), thisclass::set_write_request(); }
	virtual void set_multi_write_request(const address_type& a, size_t b)
		{ set_multi(b), thisclass::set_write_request(a); }
	virtual void set_multi_write_grant(void)
		{ set_multi(), thisclass::set_write_grant(); }
	virtual void set_multi_write_ack(void)
		{ set_multi(), thisclass::set_write_ack(); }
	virtual void set_multi_write_nack(void)
		{ set_multi(), thisclass::set_write_nack(); }
	virtual void set_multi_write_data(void)
		{ set_multi(), thisclass::set_write_data(); }
	virtual void set_multi_write_data(const data_type& a, size_t b, size_t c)
		{ set_multi(b, c), thisclass::set_write_data(a); }
};

template <class A, class D>
void bus_packet<A, D>::input(istream&)
{}

template <class A, class D>
void bus_packet<A, D>::output(ostream& os) const
{
	if (!os) return;
	const long flags = os.flags();
	switch (flags_) {
	case (read_pattern | req_pattern):
		os << "r_req(0x" << hex << address();
		if (is_multi()) os << ", *" << dec << total_packet_count();
		os << ')';
		break;
	case (write_pattern | req_pattern):
		os << "w_req(0x" << hex << address();
		if (is_multi()) os << ", *" << dec << total_packet_count();
		os << ')';
		break;
	case (read_pattern | grant_pattern):
		os << "r_grnt";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case (write_pattern | grant_pattern):
		os << "w_grnt";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case (read_pattern | ack_pattern):
		os << "r_ack";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case (write_pattern | ack_pattern):
		os << "w_ack";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case (read_pattern | nack_pattern):
		os << "r_nack";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case (write_pattern | nack_pattern):
		os << "w_nack";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case (read_pattern | data_pattern):
		os << "r_dat(0x" << hex << data();
		if (is_multi())
			os << ", " << dec
			   << packet_number() << '/' << total_packet_count();
		os << ')';
		break;
	case (write_pattern | data_pattern):
		os << "w_dat(0x" << hex << data();
		if (is_multi())
			os << ", " << dec
			   << packet_number() << '/' << total_packet_count();
		os << ')';
		break;
	default:
#ifdef DEBUG
		assert(0);
#endif
		break;
	}
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif
}

#endif /* BUS_PACKET_H */
