/*
 * <<< r3000_pipeline_stage.cc >>>
 *
 * --- R3000 pipeline state class 'r3000_pipeline_stage'
 *     Copyright (C) 1995-1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstring> // needed only for SGI C++ compiler
#include <iomanip> // needed only for SGI C++ compiler
#include <iostream>
#include <strstream>
#include "r3000_pipeline_stage.h"

inline r3000_word expand_signed(r3000_word x) {
	return (x & 0x8000UL) ? (x | 0xffff0000UL) : x;
}

template <class T>
static string to_string(const T& x)
{
	strstream buf;
	T tmp;
	if (x > 0 || x == 0) { // erase warning
		tmp = x;
	} else {
		buf << '-';
		tmp = -x;
	}
	if (tmp == 0 || ((tmp & 0xff) != 0 && ((tmp % 1000) == 0 || tmp < 1000))) {
		buf << tmp;
	} else {
		buf << hex << "0x" << tmp;
	}
	string str;
	buf >> str;
	return str;
}

r3000_pipeline_stage::register_set::register_set(void)
{}

r3000_pipeline_stage::register_set::register_set(r3000_word a, int b)
	: value(a), number(b)
{}

inline void r3000_pipeline_stage::register_set::clear(void)
{
	value = 0, number = 0;
}

r3000_pipeline_stage::r3000_pipeline_stage(void)
{
	clear();
}

r3000_pipeline_stage::r3000_pipeline_stage(const r3000_pipeline_stage& a)
	: buf(a.buf),
	  src1(a.src1),
	  src2(a.src2),
	  dst(a.dst),
	  reg_in(a.reg_in),
	  io_flags(a.io_flags),
	  acc_adr(a.acc_adr),
	  acc_dat(a.acc_dat),
	  dst_pc(a.dst_pc)
{}

r3000_pipeline_stage::~r3000_pipeline_stage()
{}

void r3000_pipeline_stage::instruction_fetch(r3000_word a, r3000_word b)
{
	buf.set(a, b);
	switch (opcode()) {
	case r3000_instruction::LB:
	case r3000_instruction::LBU:
	case r3000_instruction::LH:
	case r3000_instruction::LHU:
	case r3000_instruction::LW:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.base();
		dst.number = buf.rt();
		break;
	case r3000_instruction::LWL:
	case r3000_instruction::LWR:
		reg_in = 2;
		io_flags = dst_out;
		src1.number = buf.base();
		src2.number = buf.rt();
		dst.number = buf.rt();
		break;
	case r3000_instruction::SB:
	case r3000_instruction::SH:
	case r3000_instruction::SW:
	case r3000_instruction::SWL:
	case r3000_instruction::SWR:
		reg_in = 2;
		io_flags = 0;
		src1.number = buf.base();
		src2.number = buf.rt();
		break;
	case r3000_instruction::ADDI:
	case r3000_instruction::ADDIU:
	case r3000_instruction::SLTI:
	case r3000_instruction::SLTIU:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.rs();
		dst.number = buf.rt();
		break;
	case r3000_instruction::ANDI:
	case r3000_instruction::ORI:
	case r3000_instruction::XORI:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.rs();
		dst.number = buf.rt();
		break;
	case r3000_instruction::LUI:
		reg_in = 0;
		io_flags = dst_out;
		dst.number = buf.rt();
		break;
	case r3000_instruction::ADD:
	case r3000_instruction::ADDU:
	case r3000_instruction::SUB:
	case r3000_instruction::SUBU:
	case r3000_instruction::SLT:
	case r3000_instruction::SLTU:
	case r3000_instruction::AND:
	case r3000_instruction::OR:
	case r3000_instruction::XOR:
	case r3000_instruction::NOR:
		reg_in = 2;
		io_flags = dst_out;
		src1.number = buf.rs();
		src2.number = buf.rt();
		dst.number = buf.rd();
		break;
	case r3000_instruction::SLL:
	case r3000_instruction::SRL:
	case r3000_instruction::SRA:
		reg_in = 1;
		io_flags = dst_out;
		src1.number = buf.rt();
		dst.number = buf.rd();
		break;
	case r3000_instruction::SLLV:
	case r3000_instruction::SRLV:
	case r3000_instruction::SRAV:
		reg_in = 2;
		io_flags = dst_out;
		src1.number = buf.rt();
		src2.number = buf.rs();
		dst.number = buf.rd();
		break;
	case r3000_instruction::MULT:
	case r3000_instruction::MULTU:
	case r3000_instruction::DIV:
	case r3000_instruction::DIVU:
		reg_in = 2;
		io_flags = int_out;
		src1.number = buf.rs();
		src2.number = buf.rt();
		break;
	case r3000_instruction::MFHI:
	case r3000_instruction::MFLO:
		reg_in = 0;
		io_flags = int_in | dst_out;
		dst.number = buf.rd();
		break;
	case r3000_instruction::MTHI:
	case r3000_instruction::MTLO:
		reg_in = 1;
		io_flags = int_out;
		src1.number = buf.rs();
		break;
	case r3000_instruction::J:
		reg_in = 0;
		io_flags = int_out;
		break;
	case r3000_instruction::JAL:
		reg_in = 0;
		io_flags = int_out | dst_out;
		dst.number = 31;
		break;
	case r3000_instruction::JR:
		reg_in = 1;
		io_flags = int_out;
		src1.number = buf.rs();
		break;
	case r3000_instruction::JALR:
		reg_in = 1;
		io_flags = int_out | dst_out;
		src1.number = buf.rs();
		dst.number = buf.rd();
		break;
	case r3000_instruction::BEQ:
	case r3000_instruction::BNE:
		reg_in = 2;
		io_flags = int_out;
		src1.number = buf.rs();
		src2.number = buf.rt();
		break;
	case r3000_instruction::BLEZ:
	case r3000_instruction::BGTZ:
	case r3000_instruction::BLTZ:
	case r3000_instruction::BGEZ:
		reg_in = 1;
		io_flags = int_out;
		src1.number = buf.rs();
		break;
	case r3000_instruction::BLTZAL:
	case r3000_instruction::BGEZAL:
		reg_in = 1;
		io_flags = int_out | dst_out;
		src1.number = buf.rs();
		dst.number = 31;
		break;
	case r3000_instruction::BREAK:
	case r3000_instruction::SYSCALL:
		reg_in = 0;
		io_flags = 0;
		break;
	case r3000_instruction::LWC:
		reg_in = 1;
		io_flags = cp_out;
		src1.number = buf.base();
		break;
	case r3000_instruction::SWC:
		reg_in = 1;
		io_flags = cp_in;
		src1.number = buf.base();
		break;
	case r3000_instruction::MTC:
	case r3000_instruction::CTC:
		reg_in = 1;
		io_flags = cp_out;
		src1.number = buf.rt();
		dst.number = buf.rd();
		break;
	case r3000_instruction::MFC:
	case r3000_instruction::CFC:
		reg_in = 0;
		io_flags = cp_in | dst_out;
		src1.number = buf.rd();
		dst.number = buf.rt();
		break;
	case r3000_instruction::BCT:
	case r3000_instruction::BCF:
		reg_in = 0;
		io_flags = int_out;
		break;
	case r3000_instruction::COP:
		reg_in = 0;
		io_flags = 0;
		break;
	default:
		reg_in = 0;
		io_flags = 0;
		break;
	}
	if (is_destination_output() && destination_number() == 0)
		io_flags &= ~dst_out;
}

void r3000_pipeline_stage::execute(void)
{
	switch (buf.opcode()) {
	case r3000_instruction::LB:
	case r3000_instruction::LBU:
	case r3000_instruction::LH:
	case r3000_instruction::LHU:
	case r3000_instruction::LW:
	case r3000_instruction::LWL:
	case r3000_instruction::LWR:
		acc_adr = source1() + expand_signed(buf.immediate());
		break;
	case r3000_instruction::SB:
	case r3000_instruction::SH:
	case r3000_instruction::SW:
	case r3000_instruction::SWL:
	case r3000_instruction::SWR:
		acc_adr = source1() + expand_signed(buf.immediate());
		acc_dat = source2();
		break;
	case r3000_instruction::ADDI:
		dst.value = source1() + expand_signed(buf.immediate());
		break;
	case r3000_instruction::ADD:
		dst.value = source1() + source2();
		break;
	case r3000_instruction::ADDIU:
		dst.value = source1() + expand_signed(buf.immediate());
		break;
	case r3000_instruction::ADDU:
		dst.value = source1() + source2();
		break;
	case r3000_instruction::SUB:
		dst.value = source1() - source2();
		break;
	case r3000_instruction::SUBU:
		dst.value = source1() - source2();
		break;
	case r3000_instruction::SLTI:
		dst.value =
		  (signed_r3000_word(source1()) <
				signed_r3000_word(expand_signed(buf.immediate())))
		  ? 1 : 0;
		break;
	case r3000_instruction::SLT:
		dst.value =
			(signed_r3000_word(source1()) < signed_r3000_word(source2()))
			? 1 : 0;
		break;
	case r3000_instruction::SLTIU:
		dst.value = (source1() < expand_signed(buf.immediate())) ? 1 : 0;
		break;
	case r3000_instruction::SLTU:
		dst.value = (source1() < source2()) ? 1 : 0;
		break;
	case r3000_instruction::ANDI:
		dst.value = source1() & buf.immediate();
		break;
	case r3000_instruction::AND:
		dst.value = source1() & source2();
		break;
	case r3000_instruction::ORI:
		dst.value = source1() | buf.immediate();
		break;
	case r3000_instruction::OR:
		dst.value = source1() | source2();
		break;
	case r3000_instruction::XORI:
		dst.value = source1() ^ buf.immediate();
		break;
	case r3000_instruction::XOR:
		dst.value = source1() ^ source2();
		break;
	case r3000_instruction::LUI:
		dst.value = buf.immediate() << 16;
		break;
	case r3000_instruction::NOR:
		dst.value = ~(source1() | source2());
		break;
	case r3000_instruction::SLL:
		dst.value = (source1() << int(buf.sa() & 0x1f));
		break;
	case r3000_instruction::SLLV:
		dst.value = (source1() << int(source2() & 0x1f));
		break;
	case r3000_instruction::SRL:
		dst.value = (source1() >> int(buf.sa() & 0x1f));
		break;
	case r3000_instruction::SRLV:
		dst.value = (source1() >> int(source2() & 0x1f));
		break;
	case r3000_instruction::SRA:
		if ((source1() & 0x80000000UL) == 0) {
			dst.value = (source1() >> int(buf.sa() & 0x1f));
		} else {
			dst.value = ~((~source1()) >> int(buf.sa() & 0x1f));
		}
		break;
	case r3000_instruction::SRAV:
		dst.value = ((source1() & 0x80000000UL) == 0) ?
			(source1() >> int(source2() & 0x1f)) :
			~((~source1()) >> int(source2() & 0x1f));
		break;
	case r3000_instruction::MULT:
	case r3000_instruction::MULTU:
	case r3000_instruction::DIV:
	case r3000_instruction::DIVU:
	case r3000_instruction::MFHI:
	case r3000_instruction::MFLO:
	case r3000_instruction::MTHI:
	case r3000_instruction::MTLO:
		break;
	case r3000_instruction::J:
		dst_pc = buf.jump_address();
		break;
	case r3000_instruction::JAL:
		dst.value = buf.pc() + 8;
		dst_pc = buf.jump_address();
		break;
	case r3000_instruction::JR:
		dst_pc = source1();
		break;
	case r3000_instruction::JALR:
		dst.value = buf.pc() + 8;
		dst_pc = source1();
		break;
	case r3000_instruction::BEQ:
		dst_pc =
			(source1() == source2()) ? buf.branch_address() : (buf.pc() + 8);
		break;
	case r3000_instruction::BNE:
		dst_pc =
			(source1() != source2()) ? buf.branch_address() : (buf.pc() + 8);
		break;
	case r3000_instruction::BLEZ:
		dst_pc = (signed_r3000_word(source1()) <= 0) ?
					buf.branch_address() : (buf.pc() + 8);
		break;
	case r3000_instruction::BGTZ:
		dst_pc = (signed_r3000_word(source1()) > 0) ?
					buf.branch_address() : (buf.pc() + 8);
		break;
	case r3000_instruction::BLTZ:
		dst_pc = (signed_r3000_word(source1()) < 0) ?
					buf.branch_address() : (buf.pc() + 8);
		break;
	case r3000_instruction::BGEZ:
		dst_pc = (signed_r3000_word(source1()) >= 0) ?
					buf.branch_address() : (buf.pc() + 8);
		break;
	case r3000_instruction::BLTZAL:
		dst.value = buf.pc() + 8;
		dst_pc = (signed_r3000_word(source1()) < 0) ?
					buf.branch_address() : (buf.pc() + 8);
		break;
	case r3000_instruction::BGEZAL:
		dst.value = buf.pc() + 8;
		dst_pc = (signed_r3000_word(source1()) >= 0) ?
					buf.branch_address() : (buf.pc() + 8);
		break;
	case r3000_instruction::LWC:
	case r3000_instruction::SWC:
		acc_adr = source1() + expand_signed(buf.immediate());
		break;
	case r3000_instruction::MTC:
	case r3000_instruction::CTC:
		dst.value = source1();
		break;
	case r3000_instruction::MFC:
	case r3000_instruction::CFC:
		break;
	case r3000_instruction::BCT:
		dst_pc = ((source1() == 1) ? buf.branch_address() : (buf.pc() + 8));
		break;
	case r3000_instruction::BCF:
		dst_pc = ((source1() != 1) ? buf.branch_address() : (buf.pc() + 8));
		break;
	case r3000_instruction::SYSCALL:
	case r3000_instruction::BREAK:
	case r3000_instruction::COP:
	default:
		break;
	}
}

void r3000_pipeline_stage::load_fetch_for_big_endian(r3000_word a)
{
	r3000_word tmp;
	switch (buf.opcode()) {
	case r3000_instruction::LB:
		tmp = (a >> ((3 - int(access_address() & 0x3)) * 8)) & 0xff;
		dst.value = ((tmp & 0x80) == 0) ? tmp : (0xffffff00UL | tmp);
		break;
	case r3000_instruction::LBU:
		dst.value = (a >> ((3 - int(access_address() & 0x3)) * 8)) & 0xff;
		break;
	case r3000_instruction::LH:
		tmp = ((access_address() & 2) == 0) ? (a >> 16) : (a & 0xffffU);
		dst.value = ((tmp & 0x8000U) == 0) ? tmp : (0xffff0000UL | tmp);
		break;
	case r3000_instruction::LHU:
		dst.value = ((access_address() & 2) == 0) ? (a >> 16) : (a & 0xffffU);
		break;
	case r3000_instruction::LW:
	case r3000_instruction::LWC:
		dst.value = a;
		break;
	case r3000_instruction::LWL:
		switch (access_address() & 0x3) {
		case 0:
			dst.value = a;
			break;
		case 1:
			dst.value = (a << 8) | (source2() & 0xff);
			break;
		case 2:
			dst.value = (a << 16) | (source2() & 0xffffU);
			break;
		case 3:
			dst.value = (a << 24) | (source2() & 0xffffffUL);
			break;
		}
		break;
	case r3000_instruction::LWR:
		switch (access_address() & 0x3) {
		case 0:
			dst.value = (a >> 24) | (source2() & 0xffffff00UL);
			break;
		case 1:
			dst.value = (a >> 16) | (source2() & 0xffff0000UL);
			break;
		case 2:
			dst.value = (a >> 8) | (source2() & 0xff000000UL);
			break;
		case 3:
			dst.value = a;
			break;
		}
		break;
	default:
		break;
	}
}

void r3000_pipeline_stage::load_fetch_for_little_endian(r3000_word a)
{
	r3000_word tmp;
	switch (buf.opcode()) {
	case r3000_instruction::LB:
		tmp = (a >> (int(access_address() & 0x3) * 8)) & 0xff;
		dst.value = ((tmp & 0x80) == 0) ? tmp : (0xffffff00UL | tmp);
		break;
	case r3000_instruction::LBU:
		dst.value = (a >> (int(access_address() & 0x3) * 8)) & 0xff;
		break;
	case r3000_instruction::LH:
		tmp = ((access_address() & 2) == 0) ? (a & 0xffffU) : (a >> 16);
		dst.value = ((tmp & 0x8000U) == 0) ? tmp : (0xffff0000UL | tmp);
		break;
	case r3000_instruction::LHU:
		dst.value = ((access_address() & 2) == 0) ? (a & 0xffffU) : (a >> 16);
		break;
	case r3000_instruction::LW:
	case r3000_instruction::LWC:
		dst.value = a;
		break;
	case r3000_instruction::LWL:
		switch (access_address() & 0x3) {
		case 0:
			dst.value = (a << 24) | (source2() & 0xffffffUL);
			break;
		case 1:
			dst.value = (a << 16) | (source2() & 0xffffU);
			break;
		case 2:
			dst.value = (a << 8) | (source2() & 0xff);
			break;
		case 3:
			dst.value = a;
			break;
		}
		break;
	case r3000_instruction::LWR:
		switch (access_address() & 0x3) {
		case 0:
			dst.value = a;
			break;
		case 1:
			dst.value = (a >> 8) | (source2() & 0xff000000UL);
			break;
		case 2:
			dst.value = (a >> 16) | (source2() & 0xffff0000UL);
			break;
		case 3:
			dst.value = (a >> 24) | (source2() & 0xffffff00UL);
			break;
		}
		break;
	default:
		break;
	}
}

void r3000_pipeline_stage::partial_word_store_fetch_for_big_endian
	(r3000_word tmp1)
{
	r3000_word tmp2 = access_data();
	switch (opcode()) {
	case r3000_instruction::SB:
		switch (access_address() & 0x3) {
		case 0:
			acc_dat = (tmp1 & 0x00ffffffUL) | (tmp2 << 24);
			break;
		case 1:
			acc_dat = (tmp1 & 0xff00ffffUL) | ((tmp2 & 0xff) << 16);
			break;
		case 2:
			acc_dat = (tmp1 & 0xffff00ffUL) | ((tmp2 & 0xff) << 8);
			break;
		case 3:
			acc_dat = (tmp1 & 0xffffff00UL) | (tmp2 & 0xff);
			break;
		}
		break;
	case r3000_instruction::SH:
		if ((access_address() & 0x2) == 0) {
			acc_dat = (tmp1 & 0x0000ffffUL) | (tmp2 << 16);
		} else {
			acc_dat = (tmp1 & 0xffff0000UL) | (tmp2 & 0xffffU);
		}
		break;
	case r3000_instruction::SWL:
		switch (access_address() & 0x3) {
		case 0:
			acc_dat = tmp2;
			break;
		case 1:
			acc_dat = (tmp1 & 0xff000000UL) | (tmp2 >> 8);
			break;
		case 2:
			acc_dat = (tmp1 & 0xffff0000UL) | (tmp2 >> 16);
			break;
		case 3:
			acc_dat = (tmp1 & 0xffffff00UL) | (tmp2 >> 24);
			break;
		}
		break;
	case r3000_instruction::SWR:
		switch (access_address() & 0x3) {
		case 0:
			acc_dat = (tmp1 & 0xffffffUL) | (tmp2 << 24);
			break;
		case 1:
			acc_dat = (tmp1 & 0xffffUL) | (tmp2 << 16);
			break;
		case 2:
			acc_dat = (tmp1 & 0xffUL) | (tmp2 << 8);
			break;
		case 3:
			acc_dat = tmp2;
			break;
		}
		break;
	default:
		break;
	}
}

void r3000_pipeline_stage::partial_word_store_fetch_for_little_endian
	(r3000_word tmp1)
{
	r3000_word tmp2 = acc_dat;
	switch (buf.opcode()) {
	case r3000_instruction::SB:
		switch (access_address() & 0x3) {
		case 0:
			acc_dat = (tmp1 & 0xffffff00UL) | (tmp2 & 0xff);
			break;
		case 1:
			acc_dat = (tmp1 & 0xffff00ffUL) | ((tmp2 & 0xff) << 8);
			break;
		case 2:
			acc_dat = (tmp1 & 0xff00ffffUL) | ((tmp2 & 0xff) << 16);
			break;
		case 3:
			acc_dat = (tmp1 & 0xffffffUL) | (tmp2 << 24);
			break;
		}
		break;
	case r3000_instruction::SH:
		if ((access_address() & 0x2) == 0) {
			acc_dat = (tmp1 & 0xffff0000UL) | (tmp2 & 0xffffU);
		} else {
			acc_dat = (tmp1 & 0x0000ffffUL) | (tmp2 << 16);
		}
		break;
	case r3000_instruction::SWL:
		switch (access_address() & 0x3) {
		case 0:
			acc_dat = (tmp1 & 0xffffff00UL) | (tmp2 >> 24);
			break;
		case 1:
			acc_dat = (tmp1 & 0xffff0000UL) | (tmp2 >> 16);
			break;
		case 2:
			acc_dat = (tmp1 & 0xff000000UL) | (tmp2 >> 8);
			break;
		case 3:
			acc_dat = tmp2;
			break;
		}
		break;
	case r3000_instruction::SWR:
		switch (access_address() & 0x3) {
		case 0:
			acc_dat = tmp2;
			break;
		case 1:
			acc_dat = (tmp1 & 0xffUL) | (tmp2 << 8);
			break;
		case 2:
			acc_dat = (tmp1 & 0xffffUL) | (tmp2 << 16);
			break;
		case 3:
			acc_dat = (tmp1 & 0xffffffUL) | (tmp2 << 24);
			break;
		}
		break;
	default:
		break;
	}
}

void r3000_pipeline_stage::clear(void)
{
	instruction_fetch(0, 0);
	src1.clear();
	src2.clear();
	dst.clear();
	reg_in = io_flags = 0;
	acc_adr = acc_dat = dst_pc = 0;
}

void r3000_pipeline_stage::output(ostream& os) const
{
	os << buf;
	int src1_out = (reg_in >= 1 && source1_number() != 0);
	int src2_out = (reg_in >= 2 && source2_number() != 0);
	if (src1_out == 0 && src2_out == 0 &&
		(!is_destination_output() || dst.number == 0) &&
		!is_load_or_store()) {
		return;
	}
	const long flags = os.flags();
	os << " -";
	if (src1_out) {
		os << " s1(" << r3000_register_file::register_name(source1_number())
		   << "):" << to_string(source1());
	}
	if (src2_out) {
		os << " s2(" << r3000_register_file::register_name(source2_number())
		   << "):" << to_string(source2());
	}
	if (is_destination_output() && dst.number != 0) {
		os << " d("
		   << r3000_register_file::register_name(destination_number())
		   << "):" << to_string(destination());
	}
	if (is_load()) {
		os << " adr:0x" << hex << access_address();
	} else if (is_store()) {
		os << " *0x" << hex << word_aligned_access_address()
		   << '=' << to_string(access_data());
	}
	os.flags(flags);
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
}

ostream& operator<<(ostream& os, const r3000_pipeline_stage& a)
{
	if (os) a.output(os);
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
	return os;
}
