/*
 * <<< r3000_processing_element_test.cc >>>
 *
 * --- Test program for r3000_processing_element
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
#include "isis.h"
#include "r3000_processing_element.h"
#include "cui_debugger.h"

int main(int, char** argv)
{
	typedef r3000_processing_element processing_element_type;
	typedef processing_element_type::size_type size_type;
	const r3000_word local_memory_address = 0;
	const r3000_word local_memory_size = 0x20000000;
	size_type icache_size = 0x2000, dcache_size = 0x2000;
	unsigned int wait_time = 0, verbose_level = 0;
	bool icache_flag = false, dcache_flag = false, fpa_flag = false,
		 split_flag = false, clock_flag = false, trace_flag = false,
		monitor_flag = false;

	// read commandline argument for simulator
	while (*++argv != NULL && **argv == '-') {
		while (*++*argv != '\0') {
			bool flag = 0;
			switch (**argv) {
			case 'c': clock_flag = true; break;
			case 'f': fpa_flag = true; break;
			case 'm': monitor_flag = true; break;
			case 's': split_flag = true; break;
			case 't': trace_flag = clock_flag = true; break;
			case 'v': verbose_level++; clock_flag = true; break;
			case 'w': wait_time = atoi(*argv + 1); flag = true; break;
			case 'd':
				dcache_flag = true;
				if (isdigit(*(*argv + 1))) {
					dcache_size = atoi(*argv + 1);
					flag = true;
				}
				break;
			case 'i':
				icache_flag = true;
				if (isdigit(*(*argv + 1))) {
					icache_size = atoi(*argv + 1);
					flag = true;
				}
				break;
			default: break;
			}
			if (flag) break;
		}
	}
	if (*argv == NULL) {
		cerr << "No executable file specified." << endl;
		return 1;
	}

	// setup simulation conditions
	processing_element_type pe;
	pe.set_local_memory_area(local_memory_address, local_memory_size);
	if (!pe.load(*argv)) {
		cerr << *argv << ": No such file or directory." << endl;
		return 1;
	}
	if (icache_flag) pe.set_instruction_cache_size(dcache_size, 0x10);
	if (dcache_flag) pe.set_data_cache_size(icache_size, 0x10);
	if (fpa_flag) pe.connect_coprocessor();
	if (split_flag) pe.enable_split_transaction();
	if (wait_time > 0) {
		pe.set_local_memory_read_wait(wait_time);
		pe.set_local_memory_write_wait(wait_time);
	}
	pe.set_file_table_size(16);
	pe.set_standard_input_stream(cin);
	pe.set_standard_output_stream(cout);
	pe.set_standard_error_stream(cerr);
	pe.set_commandline_argument((const char* const*)(argv));

	// show simulation conditions
	if (verbose_level >= 1) {
		cout << "processor:   r3000" << endl;
		cout << "coprocessor: " << (fpa_flag ? "yes" : "no") << endl;
		cout << "icache:      ";
		if (icache_flag) {
			cout << "0x" << hex << icache_size << dec << endl;
		} else {
			cout << "none" << endl;
		}
		cout << "dcache:      ";
		if (dcache_flag) {
			cout << "0x" << hex << dcache_size << dec << endl;
		} else {
			cout << "none" << endl;
		}
		cout << "memory:      0x" << hex << setw(8) << setfill('0')
			 << local_memory_address << " - 0x" << setw(8) << setfill('0')
			 << local_memory_address + local_memory_size - 1 << dec << endl;
		cout << "memory wait: " << wait_time << "clk" << endl;
		cout << "split-trans: " << (split_flag ? "yes" : "no") << endl;
		cout << "arguments:   \"";
		for (int i = 0; argv[i] != NULL; i++) {
			if (i != 0) cout << ' ';
			cout << argv[i];
		}
		cout << '"' << endl << endl;
	}

	if (!monitor_flag) {
		// execute simulation in non-interactive mode
		while (1) {
			if (verbose_level >= 2) {
				cout << "clk:" << pe.timer_clock_value() << endl
					 << pe.processor() << endl << endl;
			} else if (trace_flag) {
				cout << "0x" << hex << pe.processor().program_counter()
					 << dec << endl;
			}
			if (pe.is_halt()) break;
			if (pe.is_bus_error()) {
				cout << hex
					 << "bus error(I:0x" << pe.processor().program_counter()
					 << ", D:0x" << pe.bus_error_address() << ')'
					 << dec << endl;
				break;
			}
			pe.clock_in();
			pe.clock_out();
		}
		if (clock_flag) cout << "clock: " << pe.timer_clock_value() << endl;
		return pe.commandline_status();
	} else {
		// execute simulation in interactive mode
		cui_debugger<processing_element_type> mon;
		mon.set(pe);
		mon.interactive_mode();
		return 0;
	}
	// not reached
}
