/*
 * <<< r3010.cc >>>
 *
 * --- Copyright (C) 1996-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cctype>
#include <cstddef>
#include <cstring> // needed only for SGI C++ compiler
#include <iomanip>
#include <iostream>
#include <strstream>
#include <string>
#include "r3010_fgr.h"
#include "r3010_fcr.h"
#include "r3010_inst.h"
#include "r3010_stage.h"
#include "r3010_bus.h"

#include "r3010_add.h"
#include "r3010_ext.h"

#include "freelist.h"
#include "r3010_forward.h"
#include "forwarder.h"
#include "r3010.h"

int r3010::is_dec_string(const string& s)
{
	int len = s.length(), i = 0;
	while (i < len && isdigit(s[i])) i++;
	return (i == len);
}

int r3010::is_hex_string(const string& s)
{
	int len = s.length(), i;
	i = (len > 2 && s[0] == '0' && (s[1] == 'x' || s[1] == 'X')) ? 2 : 0;
	while (i < len && isxdigit(s[i])) i++;
	return (i == len);
}

/* --------------------------------------------------------------- */
bool
r3010::instruction_fetch_p()
{
#ifdef DEBUG
	if (debug_level() & R3010_DEBUG_STALL) {
		cout << "instruction_fetch_p() = "
			 << ((s_rd && !s_rd->have_inst()) ? "True" : "False") << endl;
	}
#endif
	return s_rd && !s_rd->have_inst();
}

void
r3010::instruction_fetch(r3000_word w)
{
	s_rd->instruction_fetch(w);
}

bool
r3010::data_fetch_p()
{
	return bus->data_fetch_p();
}

void
r3010::data_fetch(r3000_word w)
{
	bus->data_fetch(w);
}

void
r3010::clock()
{
#ifdef DEBUG
	if (debug_level() & R3010_DEBUG_TRACE) {
		cout << *this << endl;
	}
#endif

	if (s_fwb) {
		s_fwb->fwb_clock();
		if (s_fwb->fwb_go_next()) {
			delete_stage();
		}
	}
	if (s_wb) s_wb->wb_clock();
	if (s_mem) s_mem->mem_clock();
	if (s_alu) s_alu->alu_clock();

	if (_add) _add->clock();

	if (_div) _div->clock();
	if (_mul) _mul->clock();

	if (s_rd) s_rd->rd_clock();
	if (s_if) s_if->if_clock();

	if (s_if == 0) new_stage();

	forward->step();
}

bool
r3010::data_out_p()
{
	return bus->data_out_p();
}

r3000_word
r3010::data_out()
{
	return bus->data_out();
}

void
r3010::data_out_ok()
{
	bus->data_out_ok();
}

bool
r3010::fpcond() const
{
	return csr().fpcond();
}

r3000_word
r3010::get_fgr(int idx) const
{
	return _fgr->read_word(idx);
}

r3000_word
r3010::get_fcr(int idx) const
{
	return _fcr->read_word(idx);
}

void
r3010::set_fgr(int idx, r3000_word data)
{
	_fgr->write_word(idx, data);
}

void
r3010::set_fcr(int idx, r3000_word data)
{
	_fcr->write_word(idx, data);
}

// --------------------------------------------------------------------
static char *fpa_help =
"--cp1-version\n"
"	show cp1 version\n"
"--cp1-debug={VAL|fpcond,stall,forwarding,trace}\n"
"	set debug level\n\n"
"$fX[.FMT]\n"
"	show value of floating general register X as format FMT\n"
"$fcrX\n"
"	show value of floating control register X\n"
"$fX[.FMT]=VAL\n"
"	set new value VAL to floating general register X as format FMT\n"
"$fcrX=VAL\n"
"	set new value VAL to floating control register X\n"
"help\n"
"	show this messages";

void
r3010::set_args(int argc, char const * const * argv)
{
	for (int i = 1; i < argc; i++) {
		string arg(argv[i]);
		string a;
		if (check_arg(arg, "--cp1-debug=", &a)) {
			set_debug_level(a);
		} else if (arg == "--cp1-version") {
			cout << "isis fpa : " << fpa_version_string << endl;
		}
	}
}

void
r3010::set_command(const char *command)
{
	string cmd(command);

	if (command[0] == '-') { // start option
		const char *tmp_argv[3];
		tmp_argv[0] = "";
		tmp_argv[1] = command;
		tmp_argv[2] = 0;
		set_args(2, tmp_argv);
	} else if (reg_command( command )) {
		return;
	} else if (string(command) == "help") {
		cout << fpa_help << endl;
	} else {
		cerr << "cp1:(" << command << ") unknown command" << endl;
	}
}

bool
r3010::reg_command(string cmd)
{
	int st[6], len[6];
	enum { f_word, f_single, f_double } fmt = f_word;

#if 0
	// use Regex class in libg++
	static Regex reg_re( "^\\$f\\(cr\\)?\\([0-9][0-9]?\\)\\(\\.[wsd]\\)?\\([ \t]*=[ \t]*\\(.*\\)\\)?$" );
		// in perl: $reg_re = "^\$f(cr)?([0-9][0-9]?)(.[wsd])?(\s*=\s*(.*))?$";
	if (!cmd.matches(reg_re)) return false;
	for (int i = 0; i < 6; i++) {
		f[i] = reg_re.match_info(st[i], len[i], i);
	}
#else
	// not use Regex class
	{
		const int cmd_length = cmd.length();
		int p = 0;
		if (cmd_length < 3 || cmd[p] != '$' || cmd[p + 1] != 'f') {
			return false;
		}
		p += 2;
		if (p < cmd_length - 2 && (cmd[p] == 'c' || cmd[p + 1] != 'r')) {
			st[0] = p;
			len[0] = 2;
			p += 2;
		} else {
			st[0] = len[0] = 0;
		}
		if (!isdigit(cmd[p])) return false;
		st[1] = p++;
		len[1] = 1;
		if (p < cmd_length && isdigit(cmd[p])) {
			len[1]++;
			p++;
		}
		if (p < cmd_length - 2 &&
			(cmd[p + 1] == 'w' || cmd[p + 1] == 's' || cmd[p + 1] == 'd')) {
			st[2] = p;
			len[2] = 2;
			p += 2;
		} else {
			st[2] = len[2] = 0;
		}
		if (p < cmd_length) {
			st[3] = p;
			len[3] = cmd_length - p;
			while (p < cmd_length && isspace(cmd[p])) p++;
			if (p >= cmd_length - 1 || cmd[p] != '=') return false;
			p++;
			while (isspace(cmd[p])) p++;
			st[4] = p;
			len[4] = cmd_length - p;
		} else {
			st[3] = len[3] = st[4] = len[4] = 0;
		}
	}
#endif

	int regnum;

	{
		string tmp = cmd.substr(st[2], len[2]);
		istrstream is(tmp.data());
		is >> regnum;
	}

	if (len[3] > 0) {
		string format(cmd.substr(st[3], len[3]));
		if (format == ".w") {
			fmt = f_word;
		} else if (format == ".s") {
			fmt = f_single;
		} else if (format == ".d") {
			fmt = f_double;
		}
	}

	if (len[4] > 0) { // set values
		string val = cmd.substr(st[5],len[5]);
		if (len[1] > 0) { // FCR
			r3000_word w;
			if (!is_dec_string(val)) {
				istrstream(val.data()) >> w;
			} else {
				istrstream is(val.data());
				if (val.length() > 2 && val[0] == '0' &&
					(val[1] == 'x' || val[1] == 'X')) {
					char c;
					is.get(c);
					is.get(c);
				}
				is >> hex >> w >> dec;
			}
			fcr().write_word(regnum, w);
		} else {
			r3000_word w;
			bool hex_num = false;
			if (is_hex_string(val)) {
				istrstream is(val.data());
				if (val.length() > 2 && val[0] == '0' &&
					(val[1] == 'x' || val[1] == 'X')) {
					char c;
					is.get(c);
					is.get(c);
				}
				is >> hex >> w >> dec;
				hex_num = true;
			}
			switch (fmt) {
			case f_word:
				if (!hex_num) {
					istrstream(val.data()) >> w;
				}
				fgr().write_word(regnum,w);
				break;
			case f_single:
				if (hex_num) {
					fgr().write_single(regnum,(s_float)w);
				} else {
					s_float s;
					istrstream is(val.data());
					is >> s;
					fgr().write_single(regnum,s);
				}
				break;
			case f_double:
				if (hex_num) {
					fgr().write_double(regnum,(d_float)w);
				} else {
					d_float d;
					istrstream(val.data()) >> d;
					fgr().write_double(regnum, d);
				}
				break;
			}
		}
	} else { // show
		if (len[1] > 0) { // FCR
			cout << hex << "0x" << fcr().read_word(regnum) << dec << endl;
		} else {
			switch (fmt) {
			case f_word:
				cout << hex << "0x" << fgr().read_word(regnum) << dec << endl;
				break;
			case f_single:
				cout << fgr().read_single( regnum ) << "S" << endl;
				break;
			case f_double:
				cout << fgr().read_double( regnum ) << "D" << endl;
				break;
			}
		}
	}
	// not reached
	return 0;
}

void
r3010::set_debug_level(string& val)
{
	static char *deb_args[] = {
		"fpcond",
		"stall",
		"forwarding",
		"trace",
	};

	_debug_level = 0;
	if (val == "") return;

	if (val[0] == '-' || isdigit(val[0])) {
		istrstream(val.data()) >> _debug_level;
	} else {
		string v[16];
		size_t i, n, p;
		for (p = i = 0; i < sizeof(v) / sizeof(v[0]); i++) {
			while (p < val.length() && (val[p] == ' ' || val[p] == '\t')) p++;
			if (p == val.length()) break;
			v[i] = "";
			while (p < val.length() && val[p] != ' ' &&
				   val[p] != '\t' && val[p] != ',') {
				v[i] += val[p++];
			}
			if (p == val.length()) break;
			while (p < val.length() && (val[p] == ' ' || val[p] == '\t')) p++;
			if (p == val.length()) break;
			if (val[p] == ',') p++;
		}
		n = i;

		for (i = 0; i < n; i++) {
			for (unsigned j = 0;
				 j < sizeof(deb_args) / sizeof(deb_args[0]);
				 j++) {
				if (v[i] == deb_args[j]) {
					_debug_level |= (1<<j);
					break;
				}
			}
		}
	}
}

bool
r3010::check_arg(string key, const char* before, string *val)
{
	if (key.find(before) == 0) {
		*val = key.substr(key.find(before));
		return true;
	}
	return false;
}

ostream& operator<<( ostream& os, const r3010& ex ) {
	return os << (r3010&)ex;
}
