/*
 * <<< r3010_base.cc >>>
 *
 * --- Copyright (C) 1996-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstring> // needed only for SGI C++ compiler
#include <iomanip>
#include <iostream>
#include "r3010_fgr.h"
#include "r3010_fcr.h"
#include "r3010_inst.h"
#include "r3010_stage.h"
#include "r3010_bus.h"

#include "r3010_add.h"
#include "r3010_ext.h"

#include "r3010_forward.h"
#include "forwarder.h"
#include "freelist.h"
#include "r3010_base.h"

freelist<void*> forward_elem<r3010_forward>::free_forward_elem;
freelist<void*> forward_list<r3010_forward>::free_forward_list;

r3010_base::r3010_base()
{
	s_if = s_rd = s_alu = s_mem = s_wb = s_fwb = 0;
	freestage = new freelist<r3010_stage*>;

	forward = new forwarder<r3010_forward>;

	_fgr = new r3010_fgrbuf;
	_fcr = new r3010_fcrbuf( this );
	_csr = _fcr->csr();

	bus = new r3010_bus;

	_add = new r3010_add( this );
	_mul = new r3010_mul( this );
	_div = new r3010_div( this );

	_debug_level = 0;
	reset();
}

void
r3010_base::reset()
{
	// flush out pipe line stages
	if( s_fwb ) { freestage->put( s_fwb ); s_fwb = 0; }
	if( s_wb ) { freestage->put( s_wb ); s_wb = 0; }
	if( s_mem ) { freestage->put( s_mem ); s_mem = 0; }
	if( s_alu ) { freestage->put( s_alu ); s_alu = 0; }
	if( s_rd ) { freestage->put( s_rd ); s_rd = 0; }
	if( s_if ) { freestage->put( s_if ); s_if = 0; }

	forward->reset();
	_fgr->reset();
	_fcr->reset();

	bus->reset();

	_add->reset();
	_mul->reset();
	_div->reset();

	for( int i = 0; i < NFGR+NFCR; i++ )
		reg_lock[i] = false;

	new_stage();
	s_rd = s_if;
	new_stage();
}

void
r3010_base::new_stage()
{
	if( !(s_if = freestage->get()) )
		s_if = new r3010_stage( this );
	s_if->reset();
}

void
r3010_base::delete_stage()
{
	freestage->put( s_fwb );
	s_fwb = 0;
}

bool
r3010_base::lock_reg_p( int regnum )
{
	return reg_lock[regnum];
}

void
r3010_base::lock_reg( int regnum )
{
	reg_lock[regnum] = true;
}

void
r3010_base::release_reg( int regnum )
{
	reg_lock[regnum] = false;
}

r3010_add&
r3010_base::ex_add()
{
	return *_add;
}

r3010_mul&
r3010_base::ex_mul()
{
	return *_mul;
}

r3010_div&
r3010_base::ex_div()
{
	return *_div;
}

/* --------------------------------------------------------------- */
void
r3010_base::show_all_fgr( ostream& os ) const
{
	int i;

	for( i = 0; i < NFGR; i += 4 ) {
		os << "$f[" << i << "," << i+1 << "]=" << _fgr->reg(i) << "\t";
		os << "$f[" << i+2 << "," << i+3 << "]=" << _fgr->reg(i+2) << endl;
	}
}

void
r3010_base::show_all_fcr( ostream& os ) const
{
	for( int i = 0; i < NFCR; i += 4 ) {
		for( int j = 0; j < 4; j++ ) {
			os << setw(2) << (i+j) << ":0x"
			   << hex << setw(8) << setfill('0')
			   << fcr().read_word(i+j)
			   << dec << " ";
		}
		os << endl;
	}
}

ostream&
operator<<( ostream& os, const r3010_base& fpa )
{
	os << "FGR:\n";
	fpa.show_all_fgr( os );
	os << "FCR:\n";
	fpa.show_all_fcr( os );
	if( fpa.s_if )  os << *fpa.s_if;  else os << "IF:*nil*";
	os << "\n"; if( fpa.s_rd )  os << *fpa.s_rd;  else os << "RD:*nil*";
	os << "\n"; if( fpa.s_alu ) os << *fpa.s_alu; else os << "ALU:*nil*";
	os << "\n"; if( fpa.s_mem ) os << *fpa.s_mem; else os << "MEM:*nil*";
	os << "\n"; if( fpa.s_wb )  os << *fpa.s_wb;  else os << "WB:*nil*";
	os << "\n"; if( fpa.s_fwb ) os << *fpa.s_fwb; else os << "FWB:*nil*";
	os << "\n"; os << "forwarding path=" << *fpa.forward;
	os << "\n"; os << "bus=" << *fpa.bus;

	os << "\nadder = " << *fpa._add;
	os << "\nmul. = " << *fpa._mul;
	os << "\ndiv. = " << *fpa._div << "\n";

	return os;
}
