/* -*- C++ -*-
 *
 * <<< cyclic_queue.h >>>
 *
 * --- Cyclic queue class 'cyclic_queue'
 *     Copyright (C) 1995-1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef CYCLIC_QUEUE_H
#define CYCLIC_QUEUE_H 1

#ifdef DEBUG
# include <cassert>
#endif // DEBUG
#include <vector>

template < class T, class C = vector<T> >
class cyclic_queue
{
public:
	typedef C container_type;
	typedef typename C::value_type value_type;
	typedef typename C::size_type size_type;
	typedef typename C::reference reference;
	typedef typename C::const_reference const_reference;
protected:
	container_type c;
private:
	typedef cyclic_queue<T, C> thisclass;
	size_type front_, back_, size_;
public:
	cyclic_queue(void)
		: c(), front_(0), back_(0), size_(0) {}
	explicit cyclic_queue(size_type a)
		: c(a), front_(0), back_(0), size_(0) {}
	cyclic_queue(const thisclass& a)
		: c(a.c), front_(a.front_), back_(a.back_), size_(a.size_) {}
	bool empty(void) const { return size_ == 0; }
	bool full(void) const { return size_ == c.size(); }
	size_type size(void) const { return size_; }
	size_type max_size(void) const { return c.size(); }
	inline const_reference front(void) const;
	inline reference front(void);
	inline void push(const_reference);
	inline void pop(void);
	inline void reserve(size_type);
	void clear(void) { front_ = back_ = size_ = 0; }
	const_reference operator[](size_type i) const
		{ return c[(front_ + i < c.size()) ?
				   front_ + i : front_ + i - c.size()]; }
	reference operator[](size_type i)
		{ return c[(front_ + i < c.size()) ?
				   front_ + i : front_ + i - c.size()]; }
};

template <class T, class C>
inline cyclic_queue<T, C>::const_reference cyclic_queue<T, C>::front(void) const
{
#ifdef DEBUG
	assert(size_ > 0);
#endif
	return c[front_];
}

template <class T, class C>
inline cyclic_queue<T, C>::reference cyclic_queue<T, C>::front(void)
{
#ifdef DEBUG
	assert(size_ > 0);
#endif
	return c[front_];
}

template <class T, class C>
inline void cyclic_queue<T, C>::push(cyclic_queue<T, C>::const_reference a)
{
#ifdef DEBUG
	assert(size_ < max_size());
#endif
	c[back_] = a;
	size_++;
	if (++back_ == c.size()) back_ = 0;
}

template <class T, class C>
inline void cyclic_queue<T, C>::pop(void)
{
#ifdef DEBUG
	assert(size_ > 0);
#endif
	size_--;
	if (++front_ == c.size()) front_ = 0;
}

template <class T, class C>
inline void cyclic_queue<T, C>::reserve(cyclic_queue<T, C>::size_type a)
{
	const size_type old_capacity = max_size(), new_capacity = a;
	if (new_capacity == old_capacity) return;
	thisclass tmp(a);
	while (!empty()) {
		tmp.push(front());
		pop();
	}
	c.resize(a);
	clear();
	while (!tmp.empty()) {
		push(tmp.front());
		tmp.pop();
	}
}

#endif /* CYCLIC_QUEUE_H */
