/* -*- C++ -*-
 *
 * <<< integer_hash_map.h >>>
 *
 * --- Hash_map class only for integer_type 'integer_hash_map'
 *     Copyright (C) 1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef INTEGER_HASH_MAP_H
#define INTEGER_HASH_MAP_H 1

#include <algorithm>
#include <iostream>
#include <iomanip>
#include <vector>

template <class K, class T>
class integer_hash_map
{
public:
	typedef K key_type;
	typedef T mapped_type;
	typedef size_t size_type;
private:
	typedef integer_hash_map<K, T> thisclass;
	static const size_type default_hash_size = 4093; // prime number
	struct entry_type {
		key_type key;
		mapped_type val;
		entry_type* next;
		entry_type(key_type a) : key(a), val(), next(NULL) {}
		entry_type(const entry_type& a) : key(a.key), val(a.val), next(NULL) {}
	};
	typedef vector<entry_type*> hash_table_type;
	hash_table_type buf;
	size_type size_, hash_size_;
	size_type hash_function(key_type a) const { return a % hash_size_; }
	void append_entry(size_type a, entry_type* b)
	{
		entry_type* p = buf[a];
		if (p == NULL) {
			buf[a] = b;
		} else {
			while (p->next != NULL) p = p->next;
			p->next = b;
		}
	}
	void assign(const thisclass&);
public:
	integer_hash_map(void);
	explicit integer_hash_map(const thisclass&);
	~integer_hash_map();
	thisclass& operator=(const thisclass&);
	void resize(size_type);
	void clear(void);
	size_type size(void) const { return size_; }
	size_type bucket_count(void) const { return hash_size_; }
	inline size_type count(key_type) const;
	inline const mapped_type* search(key_type) const;
	inline mapped_type* search(key_type);
	inline mapped_type& operator[](key_type);
};

template <class K, class T>
inline integer_hash_map<K, T>::size_type integer_hash_map<K, T>::count
	(integer_hash_map<K, T>::key_type i) const
{
	entry_type* p = buf[hash_function(i)];
	if (p == NULL) return 0;
	do {
		if (p->key == i) return 1;
		p = p->next;
	} while (p != NULL);
	return 0;
}

template <class K, class T>
inline const integer_hash_map<K, T>::mapped_type* integer_hash_map<K, T>::search
	(integer_hash_map<K, T>::key_type i) const
{
	size_type idx = hash_function(i);
	const entry_type* p = buf[idx];
	if (p != NULL) {
		while (1) {
			if (p->key == i) return &(p->val);
			if (p->next == NULL) break;
			p = p->next;
		}
	}
	return NULL;
}

template <class K, class T>
inline integer_hash_map<K, T>::mapped_type* integer_hash_map<K, T>::search
	(integer_hash_map<K, T>::key_type i)
{
	size_type idx = hash_function(i);
	entry_type* p = buf[idx];
	if (p != NULL) {
		while (1) {
			if (p->key == i) return &(p->val);
			if (p->next == NULL) break;
			p = p->next;
		}
	}
	return NULL;
}

template <class K, class T>
inline integer_hash_map<K, T>::mapped_type& integer_hash_map<K, T>::operator[]
	(integer_hash_map<K, T>::key_type i)
{
	size_type idx = hash_function(i);
	entry_type* p = buf[idx];
	if (p != NULL) {
		while (1) {
			if (p->key == i) return p->val;
			if (p->next == NULL) break;
			p = p->next;
		}
		// entry is not found
		entry_type* t = new entry_type(i);
		p->next = t;
		size_++;
		return t->val;
	} else {
		// hash-entry is not found
		entry_type* t = new entry_type(i);
		buf[idx] = t;
		size_++;
		return t->val;
	}
	// not reached
}

template <class K, class T>
void integer_hash_map<K, T>::assign(const integer_hash_map<K, T>& a)
{
	for (size_type i = 0; i < a.hash_size_; i++) {
		if (a.buf[i] == NULL) continue;
		entry_type* p = a.buf[i];
		while (1) {
			entry_type* t = new entry_type(*p);
			append_entry(hash_function(p->key), t);
			if (p->next == NULL) break;
			p = p->next;
		}
	}
}

template <class K, class T>
integer_hash_map<K, T>::integer_hash_map(void)
	: buf(default_hash_size, NULL),
	  size_(0),
	  hash_size_(default_hash_size)
{}

template <class K, class T>
integer_hash_map<K, T>::integer_hash_map(const integer_hash_map<K, T>& a)
	: buf(a.hash_size_),
	  size_(a.size_),
	  hash_size_(a.hash_size_)
{
	fill(buf.begin(), buf.end(), (typename hash_table_type::value_type)(NULL));
	assign(a);
}

template <class K, class T>
integer_hash_map<K, T>::~integer_hash_map()
{
	clear();
}

template <class K, class T>
integer_hash_map<K, T>& integer_hash_map<K, T>::operator=
	(const integer_hash_map<K, T>& a)
{
	if (this != &a) {
		clear();
		assign(a);
		size_ = a.size_;
	}
	return *this;
}

template <class K, class T>
void integer_hash_map<K, T>::resize(integer_hash_map<K, T>::size_type a)
{
	hash_table_type tmp_buf(buf);
	buf.resize(a);
	hash_size_ = a;
	fill(buf.begin(), buf.end(), (typename hash_table_type::value_type)(NULL));
	for (size_type i = 0; i < tmp_buf.size(); i++) {
		if (tmp_buf[i] == NULL) continue;
		entry_type* p = tmp_buf[i];
		while (1) {
			entry_type* n = p->next;
			p->next = NULL;
			append_entry(hash_function(p->key), p);
			if (n == NULL) break;
			p = n;
		}
	}
}

template <class K, class T>
void integer_hash_map<K, T>::clear(void)
{
	for (size_type i = 0; i < buf.size(); i++) {
		entry_type* p = buf[i];
		while (p != NULL) {
			entry_type* n = p->next;
			delete p;
			p = n;
		}
	}
	fill(buf.begin(), buf.end(), (typename hash_table_type::value_type)(NULL));
	size_ = 0;
}

#endif /* INTEGER_HASH_MAP_H */
