/* -*- C++ -*-
 * Copyright (c) 1995-1996 Tohru Kisuki
 *               1995-1998 Masaki Wakabayashi
 *               1998 Keisuke Inoue
 *      	 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */
/*
 * <<< cache_line_illinois.h >>>
 * Cache line buffer for Illinos protocol
 */

#ifndef CACHE_LINE_ILLINOIS_H
#define CACHE_LINE_ILLINOIS_H

#include <cstddef>
#include <iostream>
#include <iomanip>
#include "cache_line_base.h"

template < class A, class D = A, class C = vector<D> >
class cache_line_illinois : public cache_line_base<A, D, C> {
private:
	typedef cache_line_illinois<A, D, C> thisclass;
	typedef cache_line_base<A, D, C> inherited;
	enum state_t {I, CE, CS, DE};
	state_t state;
public:
	typedef C container_type;
	typedef typename C::size_type size_type;
	typedef A address_type;
	typedef D data_type;
	cache_line_illinois();
	explicit cache_line_illinois(size_t);
	cache_line_illinois(const thisclass&);
	~cache_line_illinois();
	void output(ostream&) const;
	//
	bool is_valid() const {
		return state != I; }
	bool is_exclusive() const {
		return state == CE || state == DE; }
	bool is_clean_exclusive() const {
		return state == CE; }
	bool is_clean_shared() const {
		return state == CS; }
	bool is_dirty() const {
		return state == DE; }
	//
	void set_invalid() {
		state = I; }
	void set_clean_exclusive() {
		state = CE; }
	void set_clean_shared() {
		state = CS; }
	void set_dirty() {
		state = DE; }
};

template <class A, class D, class C>
cache_line_illinois<A, D, C>::cache_line_illinois()
{
	set_invalid();
}

template <class A, class D, class C>
cache_line_illinois<A, D, C>::cache_line_illinois(size_t x)
{
	set_invalid();
}

template <class A, class D, class C>
cache_line_illinois<A, D, C>::cache_line_illinois
(const cache_line_illinois<A, D, C>& a)
{
	state = a.state;
}

template <class A, class D, class C>
cache_line_illinois<A, D, C>::~cache_line_illinois()
{
}

template <class A, class D, class C>
void cache_line_illinois<A, D, C>::output(ostream& os) const
{
	const char fill = os.fill();
	const long flags = os.flags();
	os << hex << setfill('0');
	if (is_valid()) {
		os << setw(sizeof(address_type) * 2) << tag() << ": ";
		switch (state) {
		case CE: os << "CE"; break;
		case CS: os << "CS"; break;
		case DE: os << "D "; break;
		default: os << "??"; break;
		}
		for (size_t i = 0; i < size(); i++) {
			os << ' ' << setw(sizeof(data_type) * 2) << (*this)[i];
		}
	} else {
		os << "<invalid>";
	}
	os.fill(fill);
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

template <class A, class D, class C>
ostream& operator<<(ostream& os, const cache_line_illinois<A, D>& a)
{
	if (os) a.output(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
	return os;
}

#endif // CACHE_LINE_ILLINOIS_H
